package ru.yandex.iex.proxy.flightextractor.flightabstructs;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;

import ru.yandex.iex.proxy.flightextractor.FlightExtractor;
import ru.yandex.iex.proxy.tickethandlerlegacy.TicketIEX;
import ru.yandex.iex.proxy.tickethandlerlegacy.TicketWidgetType;
import ru.yandex.json.writer.JsonType;

public abstract class FlightExtractorEntrails extends UndefinedDataProvider {
    private static final String TYPE_FROM_EXTRACTOR = "type_from_extractor";
    private static final String PAYMENT_URL = "payment_url";

    private InterfaceDataProvider finalExtractor;
    private Map<String, Object> iexMap = new HashMap<>();
    private ArrayList<Map<String, Object>> arrayOfIexMaps = new ArrayList<>();

    public enum SourceTypes {
        THY,
        CORAL_RU,
        BLABLACAR,
        SINDBAD,
        GO2SEE,
        SUNMAR,
        SUPERKASSA,
        PEGAST,
        BGOPERATOR,
        UNKNOWN
    }

    public FlightExtractorEntrails(
        final String domain,
        final long receivedDate,
        final String text)
    {
        initCurrentFlight(domain, receivedDate, text);
    }

    public FlightExtractorEntrails(
        final String domain,
        final long receivedDate,
        final Map<?, ?> obj)
    {
        if (obj != null) {
            for (final Map.Entry<?, ?> x : obj.entrySet()) {
                String key = (String) x.getKey();
                if (key.startsWith("flight")
                    && !key.startsWith(TicketIEX.FLIGHT_NUMBER))
                {
                    FlightExtractor f = new FlightExtractor(
                        domain,
                        receivedDate,
                        (String) x.getValue());
                    if (f.hasFlight()) {
                        arrayOfIexMaps.add(f.getCurrentFlightAsMap());
                    }
                }
            }
            if (arrayOfIexMaps.isEmpty()) {
                initCurrentFlight(domain, receivedDate, obj);
            }
        }
    }

    private void initCurrentFlight(
        final String domain,
        final long receivedDate,
        final Object obj)
    {
        AbstractExtractor extractor;
        SourceTypes type = getTypesOfSource(domain);
        extractor = getExtractor(type, receivedDate, obj);
        finalExtractor = extractor.createTypeSpecificExtractor();
        completeIexMap();
        arrayOfIexMaps.add(getCurrentFlightAsMap());
    }

    protected abstract AbstractExtractor getExtractor(
        final SourceTypes type,
        final long receivedDate,
        final Object obj);

    protected abstract SourceTypes getTypesOfSource(final String dom);

    @Override
    public String getFlightNumber() {
        return finalExtractor.getFlightNumber();
    }

    @Override
    public String getFlightDepDate() {
        return finalExtractor.getFlightDepDate();
    }

    @Override
    public String getFlightArrDate() {
        return finalExtractor.getFlightArrDate();
    }

    @Override
    public String getFlightDepCity() {
        return finalExtractor.getFlightDepCity();
    }

    @Override
    public String getFlightArrCity() {
        return finalExtractor.getFlightArrCity();
    }

    @Override
    public String getWidgetSubtype() {
        return finalExtractor.getWidgetSubtype();
    }

    @Override
    public String getOrigin() {
        return finalExtractor.getOrigin();
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("flight: ");
        sb.append(getFlightNumber());
        sb.append(" city_dep: ");
        sb.append(getFlightDepCity());
        sb.append(" city_arr: ");
        sb.append(getFlightArrCity());
        sb.append(" date_dep: ");
        sb.append(getFlightDepDate());
        sb.append(" date_arr: ");
        sb.append(getFlightArrDate());
        sb.append(" map: ");
        sb.append(JsonType.HUMAN_READABLE.toString(iexMap.toString()));
        return new String(sb);
    }

    public boolean hasFlight() {
        return arrayOfIexMaps.size() > 0;
    }

    public ArrayList<Map<String, Object>> getArrayOfFlights() {
        return arrayOfIexMaps;
    }

    protected Map<String, Object> getCurrentFlightAsMap() {
        return iexMap;
    }

    private void completeIexMap() {
        if (!getFlightNumber().isEmpty()) {
            putHook(TicketIEX.FLIGHT_NUMBER, getFlightNumber());
        }
        if (!getFlightDepDate().isEmpty()) {
            putHook(TicketIEX.DATE_DEP, getFlightDepDate());
            putHook(TicketIEX.TIME_DEP, getFlightDepDate());
        }
        if (!getFlightArrDate().isEmpty()) {
            putHook(TicketIEX.DATE_ARR, getFlightArrDate());
            putHook(TicketIEX.TIME_ARR, getFlightArrDate());
        }
        if (!getFlightDepCity().isEmpty()) {
            putHook(TicketIEX.CITY_DEP, formatCityName(getFlightDepCity()));
        }
        if (!getFlightArrCity().isEmpty()) {
            putHook(TicketIEX.CITY_ARR, formatCityName(getFlightArrCity()));
        }
        if (!iexMap.isEmpty() && !iexMap.containsKey(TicketIEX.FLIGHT_NUMBER)) {
            putHook(TicketIEX.FLIGHT_NUMBER, "undefined");
        }
        if (!getOrigin().isEmpty()) {
            putHook("origin", getOrigin());
        }
        if (!getWidgetSubtype().isEmpty()) {
            putHook(TYPE_FROM_EXTRACTOR, getWidgetSubtype());
        }
        iexMap.putAll(finalExtractor.getStub());
        Object paymentUrl = iexMap.get(PAYMENT_URL);
        if (paymentUrl instanceof String) {
            iexMap.put("url", paymentUrl);
            iexMap.remove(PAYMENT_URL);
            iexMap.put(TYPE_FROM_EXTRACTOR, TicketWidgetType.BOOKING);
        }
    }

    private void putHook(final String key, final String value) {
        if (value != null && !value.isEmpty()) {
            iexMap.put(key, value);
        }
    }

    private String formatCityName(final String city) {
        if (city != null && city.length() > 1) {
            return Character.toUpperCase(city.charAt(0))
                + city.substring(1).toLowerCase(Locale.getDefault());
        }
        return city;
    }
}
