package ru.yandex.iex.proxy.flightextractor.flightsources;

import java.util.ArrayList;

import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;

import ru.yandex.iex.proxy.XRegexpUtils;
import ru.yandex.iex.proxy.XTimeUtils;
import ru.yandex.iex.proxy.flightextractor.flightabstructs.AbstractExtractor;
import ru.yandex.iex.proxy.flightextractor.flightabstructs.InterfaceDataProvider;
import ru.yandex.iex.proxy.flightextractor.flightabstructs.UndefinedDataProvider;

/*
Input example:
ADANA/ADA ISTANBUL/IST TK 2481 J 04FEB 0050  0235  B  20 OK OPEN
 */

public class ThyExtractor extends AbstractExtractor {
    private static final int MILLIS = 1000;
    private static final DateTimeZone TIMEZONE =
        DateTimeZone.forID("Europe/Moscow");

    private String arrCity = "";
    private String depCity = "";
    private String flightNumber = "";
    private String commonDate = "";
    private String depDate = "";
    private String arrDate = "";
    private long receivedDate;
    private TokenState currentState = new StartState();
    private boolean flightFound = false;

    public ThyExtractor(final Object t, final long receivedDate) {
        super(t);
        this.receivedDate = receivedDate;
        String[] ar = text.split("\\s");
        for (String x : ar) {
            currentState.nextToken(x);
            if (flightFound) {
                break;
            }
        }
    }

    private interface TokenState {
        void nextToken(final String s);
    }

    private class StartState implements TokenState {
        public void nextToken(final String s) {
            if (isCityWord(s)) {
                depCity = takeCityOnly(s);
                setState(new DepCityState());
            }
        }
    }

    private class DepCityState implements TokenState {
        public void nextToken(final String s) {
            if (isCityWord(s)) {
                arrCity = takeCityOnly(s);
                setState(new ArrCityState());
            }
        }
    }

    private class ArrCityState implements TokenState {
        public void nextToken(final String s) {
            if (isFirstPartOfFlightNumber(s)) {
                flightNumber = s;
                setState(new FlightNumberPart1State());
            }
        }
    }

    private class FlightNumberPart1State implements TokenState {
        public void nextToken(final String s) {
            if (isNumber(s)) {
                flightNumber += s;
                setState(new FlightNumberPart2State());
            }
        }
    }

    private class FlightNumberPart2State implements TokenState {
        public void nextToken(final String s) {
            if (isDate(s)) {
                commonDate = transformToIexData(s);
                setState(new DateState());
            }
        }
    }

    private class DateState implements TokenState {
        public void nextToken(final String s) {
            if (!commonDate.isEmpty() && isNumber(s)) {
                depDate = commonDate + ' ' + transformTimeToIexTime(s);
                setState(new DepTimeState());
            }
        }
    }

    private class DepTimeState implements TokenState {
        public void nextToken(final String s) {
            if (!commonDate.isEmpty() && isNumber(s)) {
                arrDate = commonDate + ' ' + transformTimeToIexTime(s);
                setState(new ArrTimeState());
            }
        }
    }

    private class ArrTimeState implements TokenState {
        public void nextToken(final String s) {
            flightFound = true;
        }
    }

    private void setState(final TokenState state) {
        currentState = state;
    }

    public InterfaceDataProvider createTypeSpecificExtractor() {
        return new ThyDataProvider();
    }

    class ThyDataProvider extends UndefinedDataProvider {
        @Override
        public String getFlightNumber() {
            return flightNumber;
        }

        @Override
        public String getFlightDepDate() {
            return depDate;
        }

        @Override
        public String getFlightArrDate() {
            return arrDate;
        }

        @Override
        public String getFlightDepCity() {
            return depCity;
        }

        @Override
        public String getFlightArrCity() {
            return arrCity;
        }

        @Override
        public String getOrigin() { return "THY_EXTRACTOR"; }
    }

    private boolean isCityWord(final String w) {
        return w.matches("[A-Z]*/[A-Z]*");
    }

    private boolean isFirstPartOfFlightNumber(final String w) {
        return w.matches("[A-Z]{2}");
    }

    private boolean isNumber(final String w) {
        return w.matches("[0-9]+");
    }

    private boolean isDate(final String w) {
        return w.matches("[0-9]{2}[A-Z]{3}");
    }

    private String takeCityOnly(final String c) {
        return c.replaceFirst("/[A-Z]*", "");
    }

    private String transformToIexData(final String d) {
        String result = "";
        ArrayList<Integer> nums = XRegexpUtils.getInts(d);
        ArrayList<String> words = XRegexpUtils.getEngWords(d);
        Integer month = null;
        Integer day = null;
        if (nums.size() == 1 && words.size() == 1) {
            month = XTimeUtils.getMonthNumberFromMonthNameEng(words.get(0));
            day = nums.get(0);
        }
        if (month != null && day != null) {
            try {
                DateTime receivedDateTime =
                    new DateTime(receivedDate * MILLIS, TIMEZONE)
                        .withTimeAtStartOfDay();
                int year = receivedDateTime.getYear();
                DateTime commonDate = new DateTime(year, month, day, 0, 0);
                if (commonDate.isBefore(receivedDateTime)) {
                    ++year;
                }
                result = String.valueOf(day) + '.' + month + '.' + year;
            } catch (IllegalArgumentException e) {
                result = "";
            }
        }
        return result;
    }

    private String transformTimeToIexTime(final String t) {
        String res = t;
        if (t.length() >= 2) {
            StringBuilder sb = new StringBuilder();
            sb.append(t, 0, t.length() - 2);
            sb.append(':');
            sb.append(t, t.length() - 2, t.length());
            sb.append(":00");
            res = new String(sb);
        }
        return res;
    }
}
