package ru.yandex.iex.proxy.images;

import java.util.LinkedHashMap;
import java.util.Map;

public class Image implements Comparable<Image> {
    private static final double RATIO_THRESHOLD = 4;

    private static final int ODDITY = 31;
    private final int height;
    private final int width;
    private final double ratio;
    private final long cmpValue;
    private String url;

    public Image(final int height, final int width) {
        this.height = height;
        this.width = width;
        this.cmpValue = Math.abs(height * width);
        this.ratio = ratio(width, height);
    }

    private double ratio(final int width, final int height) {
        double ratio = Float.MAX_VALUE;
        if (height > 0 && width > 0) {
            ratio = Math.max(1.0 * height / width, 1.0 * width / height);
        }

        return ratio;
    }

    public String url() {
        return url;
    }

    public Image url(final String url) {
        this.url = url;
        return this;
    }

    public int height() {
        return height;
    }

    public int width() {
        return width;
    }

    @Override
    public int compareTo(final Image o) {
        int cmp = Boolean.compare(this.decent(), o.decent());
        if (cmp != 0) {
            return cmp;
        }
        return Long.compare(cmpValue, o.cmpValue);
    }

    public boolean decent() {
        return width > 0 && height > 0 && ratio <= RATIO_THRESHOLD;
    }

    @Override
    public boolean equals(final Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }

        return this.compareTo((Image) o) == 0;
    }

    @Override
    public int hashCode() {
        int result = height;
        result = ODDITY * result + width;
        result = ODDITY * result;

        if (url != null) {
            result += url.hashCode();
        }

        return result;
    }

    public boolean isOnlyWidth() {
        return width != -1 && height == -1;
    }

    public Map<?, ?> asMap() {
        Map<String, Object> result = new LinkedHashMap<>();
        result.put("src", url);
        result.put("height", String.valueOf(height));
        result.put("width", String.valueOf(width));
        return result;
    }

    @Override
    public String toString() {
        return "Image{"
            + "height=" + height
            + ", width=" + width
            + ", ratio=" + ratio
            + ", url='" + url + '\''
            + '}';
    }
}
