package ru.yandex.iex.proxy.images;

import java.net.MalformedURLException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import org.apache.http.HttpHost;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.iex.proxy.IexProxy;
import ru.yandex.iex.proxy.ImmutableIexProxyConfig;
import ru.yandex.iex.proxy.IndexationContext;
import ru.yandex.iex.proxy.Solution;
import ru.yandex.json.xpath.JsonUnexpectedTokenException;
import ru.yandex.json.xpath.ValueUtils;
import ru.yandex.parser.uri.QueryConstructor;

public final class ImagesFetcher {
    private final Cache<String, Image> cache;
    private final AsyncClient zoraClient;
    private final ImmutableIexProxyConfig iexConfig;
    private final IexProxy proxy;

    public ImagesFetcher(
        final AsyncClient zoraClient,
        final IexProxy proxy)
    {
        this.zoraClient = zoraClient;
        this.proxy = proxy;
        this.iexConfig = proxy.config();

        cache = CacheBuilder.newBuilder()
            .maximumSize(iexConfig.zoraProxyConfig().maxCacheSize())
            .concurrencyLevel(iexConfig.workers())
            .expireAfterWrite(
                iexConfig.zoraProxyConfig().expirePeriod(),
                TimeUnit.SECONDS).build();
    }

    public void fetch(
        final Solution result,
        final IndexationContext<Solution> context)
    {
        fetch(
            result.cokemulatorSolutions(),
            context.abstractContext().session(),
            null);
    }

    @SuppressWarnings("FutureReturnValueIgnored")
    public void fetch(
        final Map<?, ?> solutions,
        final ProxySession session,
        final FutureCallback<Image> callback)
    {
        ImmutableZoraProxyClientConfig proxyClientConfig =
            iexConfig.zoraProxyConfig();

        if (proxyClientConfig.partToDownload() <= 0) {
            if (callback != null) {
                callback.completed(null);
            }

            return;
        }

        long startTs = System.currentTimeMillis();

        Collection<String> extractedUrls = new LinkedHashSet<>();

        try {
            extractedUrls.addAll(extractUrls(solutions));
        } catch (JsonUnexpectedTokenException jute) {
            session.logger().log(
                Level.WARNING,
                "Failed to extract images urls");
            if (callback != null) {
                callback.failed(jute);
            }
            return;
        }

        MultiFutureCallback<Image> mfcb =
            new MultiFutureCallback<>(
                new MailImagesAggregateCallback(callback, startTs));

        AsyncClient proxyClient = zoraClient.adjust(session.context());

        try {
            Map<HttpHost, List<String>> requestMap = new HashMap<>();
            List<Image> cached = new ArrayList<>(extractedUrls.size());

            for (String url: extractedUrls) {
                HttpHost host = null;
                try {
                    host = proxyClientConfig.resolve(url);
                } catch (MalformedURLException mue) {
                    session.logger().warning("Malformed url " + url);
                }

                if (host == null) {
                    continue;
                }

                Image image = cache.getIfPresent(url);
                if (image != null) {
                    cached.add(image);
                    continue;
                }

                List<String> hostUrls = requestMap.get(host);
                if (hostUrls == null) {
                    hostUrls = new ArrayList<>(extractedUrls.size());
                    requestMap.put(host, hostUrls);
                }

                hostUrls.add(url);

                proxy.imageDownloadRequest();
            }

            // handle cached images
            if (cached.size() > 0) {
                Image best = cached.get(0);
                for (Image image: cached) {
                    if (image.compareTo(best) > 0) {
                        best = image;
                    }
                }

                if (best.height() > 0 && best.width() > 0) {
                    mfcb.newCallback().completed(best);
                }
            }

            int urlsToFetch = 0;
            for (Map.Entry<HttpHost, List<String>> request
                : requestMap.entrySet())
            {
                QueryConstructor qc = new QueryConstructor("/image?");
                for (String url: request.getValue()) {
                    qc.append("url", url);
                }

                urlsToFetch += request.getValue().size();

                proxyClient.execute(
                    request.getKey(),
                    new BasicAsyncRequestProducerGenerator(qc.toString()),
                    ZoraProxyResultConsumerFactory.INSTANCE,
                    session.listener().createContextGeneratorFor(proxyClient),
                    new ImagesBatchLoadingCallback(
                        mfcb.newCallback(),
                        request.getValue()));
            }

            session.logger().info(
                "Fetching images " + urlsToFetch
                    + " requests " + requestMap.size()
                    + " in cache " + cached.size());
        } catch (BadRequestException bre) {
            session.logger().log(
                Level.WARNING,
                "Unable make images download request",
                bre);
        } finally {
            mfcb.done();
        }
    }

    private static List<String> extractUrls(final Object factObj)
        throws JsonUnexpectedTokenException
    {
        Map<?, ?> fact = ValueUtils.asMapOrNull(factObj);
        if (fact != null) {
            List<?> imagesUrls = ValueUtils.asListOrNull(fact.get("all_imgs"));

            if (imagesUrls != null) {
                List<String> result = new ArrayList<>(imagesUrls.size());
                for (Object item : imagesUrls) {
                    result.add(ValueUtils.asString(item));
                }

                return result;
            }
        }

        return Collections.emptyList();
    }

    private final class ImagesBatchLoadingCallback
        extends AbstractFilterFutureCallback<List<Image>, Image>
    {
        private final List<String> urls;

        private ImagesBatchLoadingCallback(
            final FutureCallback<? super Image> callback,
            final List<String> urls)
        {
            super(callback);

            this.urls = urls;
        }

        @Override
        public void completed(final List<Image> images) {
            Image best = null;
            if (images.size() > 0) {
                best = images.get(0);
                best.url(urls.get(0));
            }

            for (int i = 0; i < images.size(); i++) {
                Image current = images.get(i);
                current.url(urls.get(i));
                cache.put(current.url(), current);

                if (current.compareTo(best) > 0) {
                    best = current;
                }
            }

            if (best != null && (best.height() <= 0 && best.width() <= 0)) {
                best = null;
            }

            callback.completed(best);
        }
    }

    private final class MailImagesAggregateCallback
        extends AbstractFilterFutureCallback<List<Image>, Image>
    {
        private final long startTs;

        private MailImagesAggregateCallback(
            final FutureCallback<Image> callback,
            final long startTs)
        {
            super(callback);
            this.startTs = startTs;
        }

        @Override
        public synchronized void completed(final List<Image> images) {
            proxy.imagesMidDownloadTimes(System.currentTimeMillis() - startTs);

            Image best = null;

            for (Image current : images) {
                if (current == null) {
                    continue;
                }

                if (best == null) {
                    best = current;
                    continue;
                }

                if (current.compareTo(best) > 0) {
                    best = current;
                }
            }

            callback.completed(best);
        }
    }
}
