package ru.yandex.iex.proxy.images;

import java.net.MalformedURLException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.List;

import org.apache.http.HttpHost;

import ru.yandex.http.config.AbstractHttpTargetConfigBuilder;
import ru.yandex.http.util.HttpHostParser;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;
import ru.yandex.parser.string.CollectionParser;

public class ZoraProxyClientConfigBuilder
    extends AbstractHttpTargetConfigBuilder<ZoraProxyClientConfigBuilder>
    implements ZoraProxyClientConfig
{
    private List<HttpHost> zoraHosts;
    private double partToDownload;
    private int maxCacheSize;
    private long expirePeriod;

    public ZoraProxyClientConfigBuilder() {
        this(ZoraProxyClientConfigDefaults.INSTANCE);
    }

    public ZoraProxyClientConfigBuilder(
        final ZoraProxyClientConfig config)
    {
        super(config);

        this.zoraHosts = config.zoraHosts();
        this.partToDownload = config.partToDownload();
        this.expirePeriod = config.expirePeriod();
        this.maxCacheSize = config.maxCacheSize();
    }

    public ZoraProxyClientConfigBuilder(
        final IniConfig config)
        throws ConfigException
    {
        this(config, ZoraProxyClientConfigDefaults.INSTANCE);
    }

    public ZoraProxyClientConfigBuilder(
        final IniConfig config,
        final ZoraProxyClientConfig defaults)
        throws ConfigException
    {
        super(config, defaults);

        Collection<String> hosts =
            config.getAll(
                "hosts",
                null,
                new CollectionParser<>(String::trim, LinkedHashSet::new));

        if (hosts != null) {
            this.zoraHosts = parseHosts(hosts.iterator());
        } else {
            this.zoraHosts = new ArrayList<>(defaults.zoraHosts());
        }

        this.partToDownload =
            config.getDouble("enabled-part", defaults.partToDownload());

        this.expirePeriod =
            config.getLong("expire-period", defaults.expirePeriod());
        this.maxCacheSize =
            config.getInt("max-cache-size", defaults.maxCacheSize());
    }

    private List<HttpHost> parseHosts(final Iterator<String> it)
        throws ConfigException
    {
        List<HttpHost> zoraHosts = new ArrayList<>();

        try {
            while (it.hasNext()) {
                zoraHosts.add(
                    HttpHostParser.INSTANCE.apply(it.next()));
            }
        } catch (MalformedURLException mue) {
            throw new ConfigException("Cannot parse zora hosts", mue);
        }

        return zoraHosts;
    }

    public ZoraProxyClientConfigBuilder zoraHosts(final List<HttpHost> hosts) {
        this.zoraHosts = new ArrayList<>(hosts);
        return self();
    }

    @Override
    public List<HttpHost> zoraHosts() {
        return zoraHosts;
    }

    public ZoraProxyClientConfigBuilder zoraStringHosts(
        final List<String> hosts)
        throws ConfigException
    {
        this.zoraHosts = parseHosts(hosts.iterator());
        return self();
    }

    public ZoraProxyClientConfigBuilder partToDownload(final double part) {
        this.partToDownload = part;
        return self();
    }

    @Override
    public double partToDownload() {
        return this.partToDownload;
    }

    @Override
    public ZoraProxyClientConfigBuilder self() {
        return this;
    }

    public ZoraProxyClientConfigBuilder maxCacheSize(final int size) {
        this.maxCacheSize = size;
        return this;
    }

    @Override
    public int maxCacheSize() {
        return maxCacheSize;
    }

    public ZoraProxyClientConfigBuilder maxCacheSize(final long time) {
        this.expirePeriod = time;
        return this;
    }

    @Override
    public long expirePeriod() {
        return expirePeriod;
    }
}
