package ru.yandex.iex.proxy.move;

import java.io.IOException;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.logging.Level;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import com.google.common.collect.Sets;
import org.apache.http.Header;
import org.apache.http.HttpException;
import org.apache.http.HttpHost;
import org.apache.http.HttpStatus;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.message.BasicHeader;

import ru.yandex.client.wmi.Folders;
import ru.yandex.client.wmi.FoldersConsumerFactory;
import ru.yandex.dbfields.ChangeType;
import ru.yandex.dbfields.MailIndexFields;
import ru.yandex.dbfields.PgFields;
import ru.yandex.http.config.ImmutableURIConfig;
import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.BadResponseException;
import ru.yandex.http.util.DoubleFutureCallback;
import ru.yandex.http.util.HttpExceptionConverter;
import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.YandexHttpStatus;
import ru.yandex.http.util.nio.AsyncStringConsumerFactory;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.HeaderAsyncRequestProducerSupplier;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.http.util.nio.client.AsyncGetURIRequestProducerSupplier;
import ru.yandex.iex.proxy.AbstractCallback;
import ru.yandex.iex.proxy.AbstractContext;
import ru.yandex.iex.proxy.AbstractFilterSearchCallback;
import ru.yandex.iex.proxy.ChangeContext;
import ru.yandex.iex.proxy.ChangeHandler;
import ru.yandex.iex.proxy.IexProxy;
import ru.yandex.iex.proxy.IexProxyConfig;
import ru.yandex.iex.proxy.IndexationContext;
import ru.yandex.iex.proxy.UpdateHandler;
import ru.yandex.iex.proxy.XJsonUtils;
import ru.yandex.iex.proxy.complaints.ComplaintProcessCallback;
import ru.yandex.iex.proxy.complaints.DeleteProcessCallback;
import ru.yandex.iex.proxy.complaints.DeletesFilterSearchHandler;
import ru.yandex.iex.proxy.complaints.MailMessageContext;
import ru.yandex.iex.proxy.complaints.Sources;
import ru.yandex.iex.proxy.complaints.TraceFutureCallback;
import ru.yandex.iex.proxy.complaints.UserAction;
import ru.yandex.iex.proxy.complaints.UserActionContext;
import ru.yandex.iex.proxy.complaints.UserActionCountersUpdater;
import ru.yandex.iex.proxy.complaints.UserActionHandler;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.json.xpath.JsonUnexpectedTokenException;
import ru.yandex.json.xpath.ValueUtils;
import ru.yandex.parser.searchmap.User;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.search.prefix.LongPrefix;
import ru.yandex.search.proxy.universal.PlainUniversalSearchProxyRequestContext;
import ru.yandex.search.proxy.universal.UniversalSearchProxyRequestContext;

public enum MoveHandler implements ChangeHandler {
    INSTANCE;

    private static final String SPAM_FOLDER_NAME = "spam";
    private static final String TRASH_FOLDER_NAME = "trash";
    private static final String HIDDEN_TRASH_FOLDER_NAME = "trash";
    private static final String FALSES_FOLDER_NAME = "falses";
    private static final String SPAM_SAMPLES_FOLDER_NAME = "spam samples";
    private static final String WMI_SUFFIX = "caller=msearch";
    private static final String MDB = "mdb";
    private static final String PG = "pg";
    private static final String UID = "uid";
    private static final String MID = "mid";
    private static final String SEEN = "seen";
    private static final String CHANGED = "changed";
    private static final String SRC_FID = "src_fid";
    private static final String FID = "fid";
    private static final String NO_FACTS = "no_facts";
    public static final long COMPAINS_UID = 991949281;

    @Override
    public void handle(final ChangeContext context) {
        Sources source = MailMessageContext.source(context.json());
        String sessionKey = MailMessageContext.sessionKey(context.json());
        if (source != Sources.IMAP && sessionKey != null && !sessionKey.isEmpty()) { // SODEV-2491
            checkMovingFromSpam(context);
        } else {
            context.session().logger().warning("MoveHandler.handle: action skipped because of source="
                + source + ", session_key=" + (sessionKey == null ? null : ("'" + sessionKey + "'")));
            context.session().response(HttpStatus.SC_OK);
        }
    }

    private static String trustedKey(final ChangeContext context) {
        return trustedKey(context.uid());
    }

    private static String trustedKey(final long uid) {
        return trustedKey(Long.toString(uid));
    }

    private static String trustedKey(final String uid) {
        return "so_trusted_complainer_" + uid;
    }

    public static Set<String> extractMids(final List<Map<String, Object>> docs) {
        int size = docs.size();
        Set<String> mids = new HashSet<>(size << 1);
        for (Map<String, Object> doc : docs) {
            mids.add(doc.get(MID).toString());
        }
        return mids;
    }

    public static void addFolderNames(final List<Map<String, Object>> docs, final Folders folders) {
        for (final Map.Entry<String, List<String>> entry : folders.fids().entrySet()) {
            for (final Map<String, Object> actionInfo : docs) {
                String mid = Long.toString((Long) actionInfo.get(SRC_FID));
                if (entry.getValue().contains(mid)) {
                    actionInfo.put(MailIndexFields.FOLDER_NAME, entry.getKey());
                }
            }
        }
    }

    private void checkMovingFromSpam(final ChangeContext context) {
        IexProxy iexProxy = context.iexProxy();
        ImmutableURIConfig foldersConfig;
        AsyncClient foldersClient;
        String tvmTicket;
        if (context.corp()) {
            foldersConfig = iexProxy.config().corpFoldersConfig();
            foldersClient = iexProxy.corpFoldersClient();
            tvmTicket = context.iexProxy().corpFoldersTvm2Ticket();
        } else {
            foldersConfig = iexProxy.config().foldersConfig();
            foldersClient = iexProxy.foldersClient();
            tvmTicket = context.iexProxy().foldersTvm2Ticket();
        }
        try {
            QueryConstructor query = new QueryConstructor(
                new StringBuilder(foldersConfig.uri().toASCIIString())
                    .append(foldersConfig.firstCgiSeparator())
                    .append(WMI_SUFFIX));
            query.append(MDB, PG);
            query.append(UID, context.prefix());
            String uri = query.toString();
            context.session().logger().info("Send folders request: " + uri);
            foldersClient.execute(
                new HeaderAsyncRequestProducerSupplier(
                    new AsyncGetURIRequestProducerSupplier(uri),
                    new BasicHeader(
                        YandexHeaders.X_YA_SERVICE_TICKET,
                        tvmTicket)),
                FoldersConsumerFactory.OK,
                context.session().listener()
                    .createContextGeneratorFor(foldersClient),
                new FoldersCallback(context));
        } catch (BadRequestException | URISyntaxException e) {
            context.session().logger().log(
                Level.SEVERE,
                "Folders request building error",
                e);
            context.session().handleException(
                HttpExceptionConverter.toHttpException(e));
        }
    }

    private static String spamSampleUrl(
        final ChangeContext context,
        final Object mid)
    {
        return "spam_samples_" + context.prefix()
            + "_so_compains_" + mid;
    }

    private static class FoldersCallback
        extends AbstractProxySessionCallback<Folders>
    {
        private final ChangeContext context;

        FoldersCallback(final ChangeContext context) {
            super(context.session());
            this.context = context;
        }

        @Override
        public void completed(final Folders folders) {
            try {
                if (context.prefix() == COMPAINS_UID) {
                    compains(folders);
                } else {
                    usersFolders(folders);
                }
            } catch (Exception e) {
                context.session().logger().log(Level.SEVERE, "FoldersCallback exception: " + e, e);
                failed(e);
            }
        }

        private void compains(final Folders folders) {
            if (context.changeType() == ChangeType.MOVE_TO_TAB) {
                session.logger().info("Skipping tab moves for compains");
                session.response(HttpStatus.SC_OK);
                return;
            }
            List<String> falsesFids = folders.fids().get(FALSES_FOLDER_NAME);
            List<String> spamSamplesFids =
                folders.fids().get(SPAM_SAMPLES_FOLDER_NAME);
            List<Map<String, Object>> movedToFalses = getMovedTo(falsesFids);
            List<Map<String, Object>> movedToSpamSamples = getMovedTo(spamSamplesFids);
            List<Map<String, Object>> movedFromSpamSamples = getMovedFrom(spamSamplesFids);
            session.logger().info(
                "Folders request completed, FALSES fids: " + falsesFids
                + '(' + movedToFalses.size() + " mids), spam samples fids: "
                + spamSamplesFids + '(' + movedToSpamSamples.size()
                + " mids to spam samples, "
                + movedFromSpamSamples + " mids moved from spam samples)");
            DoubleFutureCallback<List<Void>, Object> callback =
                new DoubleFutureCallback<>(new CompainsHookCallback(context));
            if (movedToFalses.size() + movedToSpamSamples.size() > 0) {
                try {
                    //Set<String> falseMids = extractMids(movedToFalses);
                    //Set<String> spamSamplesMids =
                    //    extractMids(movedToSpamSamples);
                    new CompainsActionsFilterSearchHandler(
                        context,
                        callback.first(),
                        extractMids(movedToFalses),
                        extractMids(movedToSpamSamples))
                        .execute();
                } catch (HttpException e) {
                    failed(e);
                    return;
                }
            } else {
                session.logger().info(
                    "Nothing moved to falses or spam samples");
                callback.first().completed(Collections.emptyList());
            }
            int removedSpamSamplesCount = movedFromSpamSamples.size();
            if (removedSpamSamplesCount == 0) {
                callback.second().completed(null);
            } else {
                Object operationId = context.json().get(PgFields.OPERATION_ID);
                StringBuilderWriter sbw = new StringBuilderWriter();
                try (JsonWriter writer = JsonType.DOLLAR.create(sbw)) {
                    writer.startObject();
                    writer.key("prefix");
                    writer.value(context.prefix());
                    writer.key("docs");
                    writer.startArray();
                    for (int i = 0; i < removedSpamSamplesCount; ++i) {
                        writer.startObject();
                        writer.key("url");
                        writer.value(
                            spamSampleUrl(
                                context,
                                movedFromSpamSamples.get(i).get(MID)));
                        writer.key("spam_sample_revision");
                        writer.value(operationId);
                        writer.key("spam_sample_type");
                        writer.value("so_compains");
                        writer.endObject();
                    }
                    writer.endArray();
                    writer.endObject();
                } catch (IOException e) {
                    callback.second().failed(e);
                    return;
                }
                IexProxy iexProxy = context.iexProxy();
                BasicAsyncRequestProducerGenerator generator =
                    new BasicAsyncRequestProducerGenerator(
                        "/modify?spam-samples-removal&prefix="
                        + context.prefix()
                        + "&operation-id=" + operationId,
                        sbw.toString());
                generator.addHeader(YandexHeaders.SERVICE, context.mailSearchQueueName());

                AsyncClient producerAsyncClient =
                    iexProxy.producerAsyncClient().adjust(
                        context.session().context());

                producerAsyncClient.execute(
                    iexProxy.config().producerAsyncClientConfig().host(),
                    generator,
                    AsyncStringConsumerFactory.OK,
                    context.session().listener()
                        .createContextGeneratorFor(producerAsyncClient),
                    callback.second());
            }
        }

        private void usersFolders(final Folders folders) {
            List<String> spamFids = folders.fids().get(SPAM_FOLDER_NAME);
            List<String> trashFids = folders.fids().get(TRASH_FOLDER_NAME);
            Map<String, String> fidToName = folders.fidToName();
            session.logger().info(
                "Folders request completed, spam fids: " + spamFids + ", trash fids: " + trashFids);
            List<Map<String, Object>> movedFromSpam = getMovedFromSpam(spamFids, trashFids, fidToName);
            List<Map<String, Object>> movedToSpam = getMovedTo(spamFids);
            List<Map<String, Object>> movedToTrash = getMovedToTrash(trashFids, spamFids, false);
            addFolderNames(movedFromSpam, folders);
            addFolderNames(movedToSpam, folders);
            addFolderNames(movedToTrash, folders);
            if (movedFromSpam.isEmpty() && movedToSpam.isEmpty()) {
                session.logger().info("Nothing moved from|to spam");
                if (movedToTrash.isEmpty()) {
                    session.response(HttpStatus.SC_OK);
                    return;
                }
            } else {
                session.logger().info("Move: fromSpam: " + movedFromSpam.size()
                    + ", toSpam: " + movedToSpam.size());
                try {
                    sendTrustedRequest(movedFromSpam, movedToSpam);
                } catch (HttpException e) {
                    failed(e);
                }
            }
            if (!movedToTrash.isEmpty()) {
                session.logger().info("Move toTrash: " + movedToTrash.size());
                try {
                    processDeletes(context, movedToTrash);
                } catch (HttpException e) {
                    context.session().logger().log(Level.SEVERE, "Processing of deletes failed: " + e.toString(), e);
                    failed(e);
                }
            }
        }

        private void sendTrustedRequest(
            final List<Map<String, Object>> movedFromSpam,
            final List<Map<String, Object>> movedToSpam)
            throws HttpException
        {
            AsyncClient client =
                context.iexProxy().searchClient().adjust(
                    context.session().context());
            UniversalSearchProxyRequestContext requestContext =
                new PlainUniversalSearchProxyRequestContext(
                    new User(
                        context.iexProxy().config().factsIndexingQueueName(),
                        new LongPrefix(context.prefix())),
                    null,
                    true,
                    client,
                    context.session().logger());
            QueryConstructor query = new QueryConstructor(
                "/search-iex-proxy-complains?IO_PRIO=100");
            query.append("prefix", context.uid());
            query.append(
                "service",
                context.iexProxy().config().factsIndexingQueueName());
            query.append("json-type", "dollar");
            query.append("get", "url");
            query.append(
                "text",
                "url:" + trustedKey(context));
            context.iexProxy().sequentialRequest(
                session,
                requestContext,
                new BasicAsyncRequestProducerGenerator(query.toString()),
                context.iexProxy().almostAllFactsTimeout(),
                true,
                JsonAsyncTypesafeDomConsumerFactory.OK,
                session.listener().createContextGeneratorFor(client),
                new CheckTrustedResponseCallback(
                    context,
                    movedFromSpam,
                    movedToSpam));
        }

        @SuppressWarnings("unchecked")
        private List<Map<String, Object>> getMovedFromSpam(
            final List<String> spamFids,
            final List<String> trashFids,
            final Map<String, String> fidToName)
        {
            List<Map<String, Object>> movedFromSpam = new ArrayList<>();
            Object changed = context.json().get(CHANGED);
            if (changed instanceof List) {
                for (Object changedItem : (List<?>) changed) {
                    if (changedItem instanceof Map) {
                        Map<String, Object> changedMap = (Map<String, Object>) changedItem;
                        String srcFid = Objects.toString(changedMap.get(SRC_FID), null);
                        String fid = Objects.toString(changedMap.get(FID), null);
                        Object mid = changedMap.get(MID);
                        if (!(mid instanceof Long)) {
                            continue;
                        }
                        if ((spamFids.contains(srcFid) || trashFids.contains(srcFid))
                            && (fid == null
                                || (!spamFids.contains(fid) && !trashFids.contains(fid))))
                        {
                            // OK, this is really looks like move from spam or
                            // trash to non-spam folder
                            // Let's check this is not move to a hidden folder
                            String targetFolderName = fidToName.get(fid);
                            if (targetFolderName != null
                                && !targetFolderName.toLowerCase(Locale.ROOT).startsWith("hidden_"))
                            {
                                movedFromSpam.add(changedMap);
                            }
                        }
                    }
                }
            }
            return movedFromSpam;
        }

        @SuppressWarnings("unchecked")
        private List<Map<String, Object>> getMovedTo(final List<String> fids) {
            List<Map<String, Object>> movedTo = new ArrayList<>();
            Object changed = context.json().get(CHANGED);
            if (changed instanceof List) {
                for (Object changedItem : (List<?>) changed) {
                    if (changedItem instanceof Map) {
                        Map<String, Object> changedMap = (Map<String, Object>) changedItem;
                        Object srcFid = changedMap.get(SRC_FID);
                        Object fid = changedMap.get(FID);
                        if (changedMap.get(MID) instanceof Long
                            && fid != null && fids.contains(fid.toString())
                            && (srcFid == null || !fids.contains(srcFid.toString())))
                        {
                            movedTo.add(changedMap);
                        }
                    }
                }
            }
            return movedTo;
        }

        @SuppressWarnings("unchecked")
        private List<Map<String, Object>> getMovedFrom(final List<String> fids) {
            List<Map<String, Object>> movedTo = new ArrayList<>();
            Object changed = context.json().get(CHANGED);
            if (changed instanceof List) {
                for (Object changedItem : (List<?>) changed) {
                    if (changedItem instanceof Map) {
                        Map<String, Object> changedMap = (Map<String, Object>) changedItem;
                        Object srcFid = changedMap.get(SRC_FID);
                        Object fid = changedMap.get(FID);
                        if (srcFid != null
                            && fids.contains(srcFid.toString())
                            && (fid == null || !fids.contains(fid.toString()))
                            && changedMap.get(MID) instanceof Long)
                        {
                            movedTo.add(changedMap);
                        }
                    }
                }
            }
            return movedTo;
        }

        @SuppressWarnings("unchecked")
        private List<Map<String, Object>> getMovedToTrash(
            final List<String> trashFids,
            final List<String> spamFids,
            final Boolean seen)
        {
            List<Map<String, Object>> movedTo = new ArrayList<>();
            Object changed = context.json().get(CHANGED);
            if (changed instanceof List) {
                for (Object changedItem : (List<?>) changed) {
                    if (changedItem instanceof Map) {
                        Map<String, Object> changedMap = (Map<String, Object>) changedItem;
                        Object srcFid = changedMap.get(SRC_FID);
                        Object fid = changedMap.get(FID);
                        if (changedMap.get(MID) instanceof Long
                                && fid != null && trashFids.contains(fid.toString())
                                && (srcFid == null ||
                                !trashFids.contains(srcFid.toString()) && !spamFids.contains(srcFid.toString()))
                                && (seen == null || changedMap.get(SEEN) == seen))
                        {
                            movedTo.add(changedMap);
                        }
                    }
                }
            }
            return movedTo;
        }
    }

    private static void processDeletes(
        final AbstractContext context,
        final List<Map<String, Object>> movedToTrash)
        throws BadRequestException
    {
        context.session().logger().info("Processing delete: uid=" + context.uid());
        final UserActionContext actionContext = new UserActionContext(context, UserAction.DELETE);
        new DeletesFilterSearchHandler(
            context,
            TraceFutureCallback.wrap(
                new DeleteProcessCallback(
                    actionContext,
                    TraceFutureCallback.wrap(
                        new UserActionCountersUpdater(
                            actionContext,
                            new MultiFutureCallback<>(
                                TraceFutureCallback.wrap(
                                    new DeletesFinalizeProcessing(actionContext),
                                    actionContext,
                                    "DeletesFinalizeProcessing")),
                            Set.of(UpdateDataExecutor.SENDER_SHINGLER)),
                        context,
                        "UserActionCountersUpdate for deletes")),
                context,
                "DeleteProcess"),
            movedToTrash
        ).execute();
    }

    private static class DeletesFinalizeProcessing extends AbstractProxySessionCallback<List<String>>
    {
        private final UserActionContext context;

        DeletesFinalizeProcessing(final UserActionContext context) {
            super(context.session());
            this.context = context;
        }

        @Override
        public void completed(final List<String> unused) {
            context.stat(YandexHttpStatus.SC_OK);
            session.response(HttpStatus.SC_OK);
        }

        @Override
        public void cancelled() {
            context.stat(YandexHttpStatus.SC_REMOTE_CLOSED_REQUEST);
            session.logger().warning("DeletesFinalizeProcessing request cancelled: " + session.listener().details());
        }

        @Override
        public void failed(final Exception e) {
            if (e instanceof BadResponseException) {
                context.stat(((BadResponseException) e).statusCode());
            } else {
                context.stat(YandexHttpStatus.SC_REMOTE_CLOSED_REQUEST);
            }
            String details = session.listener().details();
            session.logger().log(
                Level.WARNING,
                "DeletesFinalizeProcessing request failed: " + details + " because of",
                e);
            HttpException ex = HttpExceptionConverter.toHttpException(e);
            ex.addSuppressed(new Exception(details));
            session.handleException(ex);
        }
    }

    private static class CheckTrustedResponseCallback
        extends AbstractProxySessionCallback<JsonObject>
    {
        private final ChangeContext context;
        private final List<Map<String, Object>> movedFromSpam;
        private final List<Map<String, Object>> movedToSpam;

        CheckTrustedResponseCallback(
            final ChangeContext context,
            final List<Map<String, Object>> movedFromSpam,
            final List<Map<String, Object>> movedToSpam)
        {
            super(context.session());
            this.context = context;
            this.movedFromSpam = movedFromSpam;
            this.movedToSpam = movedToSpam;
        }

        @Override
        public void completed(final JsonObject result) {
            try {
                JsonMap map = result.asMap();
                JsonList hitsArray = map.getList("hitsArray");
                boolean trusted = false;
                if (hitsArray.size() > 0) {
                    JsonMap hit = hitsArray.get(0).asMap();
                    String url = hit.getString("url");
                    if (url != null && url.equals(trustedKey(context))) {
                        trusted = true;
                    }
                }
                if (movedFromSpam.isEmpty() && movedToSpam.isEmpty()) {
                    TraceFutureCallback.wrap(
                        new SpamHamHooks(
                            context,
                            movedFromSpam,
                            movedToSpam),
                        context,
                        "SpamHamHooks").completed(null);
                } else {
                    Boolean sendReports =
                        (trusted || context.iexProxy().alarmSoUser(context.prefix()))
                            && movedToSpam.size() <= 1;

                    if (sendReports) {
                        context.session().logger().info(
                            "Trusted complaint detected, will send reports");
                    } else {
                        context.session().logger().info(
                            "Ignoring multi message spam report: message count="
                                + movedToSpam.size());
                    }

                    UserAction action = updateStats(trusted);

                    final UserActionContext actionContext = new UserActionContext(context, action);
                    new FilterSearchHandler(
                        context,
                        TraceFutureCallback.wrap(
                            new ComplaintProcessCallback(
                                actionContext,
                                TraceFutureCallback.wrap(
                                    new UserActionCountersUpdater(
                                        actionContext,
                                        new MultiFutureCallback<>(
                                            TraceFutureCallback.wrap(
                                                new SpamHamHooks(
                                                    context,
                                                    movedFromSpam,
                                                    movedToSpam),
                                                context,
                                                "SpamHamHooks for MultiFutureCallback"))),
                                    context,
                                    "UserActionCountersUpdate for complaints")),
                            context,
                            "ComplaintProcess"),
                        movedFromSpam,
                        movedToSpam,
                        extractMids(movedFromSpam),
                        extractMids(movedToSpam),
                        sendReports).execute();
                }
            } catch (JsonException | BadRequestException e) {
                context.session().logger().log(Level.SEVERE, "CheckTrustedResponseCallback failed: " + e, e);
                failed(e);
            }
        }

        private UserAction updateStats(final Boolean trusted) {
            if (context.changeType() == ChangeType.MOVE_TO_TAB) {
                return UserAction.OTHER;
            }
            UserAction action = UserAction.OTHER;
            if (!movedToSpam.isEmpty()) {
                action = UserAction.SPAM;
                context.iexProxy().spamComplaints(movedToSpam.size());
                if (trusted) {
                    context.iexProxy().trustedSpamComplaints(movedToSpam.size());
                }
            }
            if (!movedFromSpam.isEmpty()) {
                action = UserAction.HAM;
                context.iexProxy().hamComplaints(movedFromSpam.size());
                if (trusted || context.iexProxy().alarmSoUser(context.prefix())) {
                    context.iexProxy().trustedHamComplaints(movedFromSpam.size());
                }
            }
            return action;
        }

        // Provides docs - bsearch's updates
        private static class FilterSearchHandler extends AbstractFilterSearchCallback<UpdateDataHolder>
        {
            private final Boolean sendReports;
            private final List<Map<String, Object>> movedFromSpam;
            private final List<Map<String, Object>> movedToSpam;
            private final Set<String> hamMids;
            private final Set<String> spamMids;

            FilterSearchHandler(
                final ChangeContext context,
                final FutureCallback<List<UpdateDataHolder>> callback,
                final List<Map<String, Object>> movedFromSpam,
                final List<Map<String, Object>> movedToSpam,
                final Set<String> hamMids,
                final Set<String> spamMids,
                final Boolean sendReports)
                throws BadRequestException
            {
                super(context, callback, Sets.union(hamMids, spamMids));
                this.movedFromSpam = movedFromSpam;
                this.movedToSpam = movedToSpam;
                this.hamMids = hamMids;
                this.spamMids = spamMids;
                this.sendReports = sendReports;
            }

            @Override
            public boolean skipEmptyEntities() {
                return false;
            }

            @Override
            public AbstractCallback<UpdateDataHolder> subMessageCallback(
                final IndexationContext<UpdateDataHolder> context)
            {
                return new SingleMessageReporter<>(context);
            }

            @Override
            public void executeSubCallback(final AbstractCallback<UpdateDataHolder> callback) {
                UserAction action = UserAction.OTHER;
                Map<?, ?> actionInfo = null;
                final Long mid = Long.parseLong(callback.context().mid());
                if (hamMids.contains(mid.toString())) {
                    action = UserAction.HAM;
                    for (final Map<?, ?> ai : movedFromSpam) {
                        if (ai.get(MID).equals(mid)) {
                            actionInfo = ai;
                            break;
                        }
                    }
                } else if (spamMids.contains(mid.toString())) {
                    action = UserAction.SPAM;
                    for (final Map<?, ?> ai : movedToSpam) {
                        if (ai.get(MID).equals(mid)) {
                            actionInfo = ai;
                            break;
                        }
                    }
                }
                UserActionHandler.requestTikaite(
                    context,
                    "/headers",
                    List.of("stid=" + callback.context().stid()),
                    TraceFutureCallback.wrap(
                        new SingleMessageHeadersCallback(
                            callback,
                            (ChangeContext) context,
                            callback.context(),
                            action,
                            actionInfo,
                            sendReports),
                        context,
                        "SingleMail"));
            }

            @Override
            public boolean skipSpam() {
                return false;
            }
        }
    }

    private static class SpamHamHooks
        extends AbstractProxySessionCallback<List<String>>
    {
        private final ChangeContext context;
        private final List<Map<String, Object>> movedFromSpam;
        private final List<Map<String, Object>> movedToSpam;

        SpamHamHooks(
            final ChangeContext context,
            final List<Map<String, Object>> movedFromSpam,
            final List<Map<String, Object>> movedToSpam)
        {
            super(context.session());
            this.context = context;
            this.movedFromSpam = movedFromSpam;
            this.movedToSpam = movedToSpam;
        }

        @Override
        public void completed(final List<String> unused) {
            if (movedFromSpam.isEmpty()) {
                session.logger().info("Nothing moved from spam");
                session.response(HttpStatus.SC_OK);
                return;
            }
            try {
                XJsonUtils.pushToMap(context.json(), CHANGED, movedFromSpam);
                List<Long> mids = new ArrayList<>();
                for (Map<?, ?> movedItem: movedFromSpam) {
                    mids.add((Long) movedItem.get(MID));
                }
                session.logger().log(
                    Level.INFO,
                    "Moved from spam, uid: " + context.prefix()
                    + ", mids: " + mids + ", initiate facts extracting");
                deleteNoFacts(mids);
            } catch (JsonUnexpectedTokenException e) {
                session.logger().log(
                    Level.SEVERE,
                    "Replacing changed failed for mids moved from spam",
                    e);
                session.response(HttpStatus.SC_BAD_REQUEST);
            }
        }

        @Override
        public void cancelled() {
            session.logger().warning("SpamHamHooks request cancelled: " + session.listener().details());
        }

        @Override
        public void failed(final Exception e) {
            String details = session.listener().details();
            session.logger().log(
                Level.WARNING,
                "SpamHamHooks request failed: " + details + " because of",
                e);
            HttpException ex = HttpExceptionConverter.toHttpException(e);
            ex.addSuppressed(new Exception(details));
            session.handleException(ex);
        }

        private void deleteNoFacts(final List<Long> mids) {
            try {
                IexProxyConfig config = context.iexProxy().config();
                String url = getUrl();
                String body = getBody(mids);
                BasicAsyncRequestProducerGenerator generator =
                    new BasicAsyncRequestProducerGenerator(url, body);
                generator.addHeader(
                    YandexHeaders.SERVICE,
                    config.factsIndexingQueueName());
                generator.addHeader(
                    YandexHeaders.X_INDEX_OPERATION_TIMESTAMP,
                    Long.toString(context.operationDateMillis()));
                String xIndexOperationQueueName = null;
                if (context.isTteot()) {
                    xIndexOperationQueueName = context.iexProxy().
                        xIndexOperationQueueNameBacklog();
                }
                if (context.zooQueueIsIexUpdate()) {
                    xIndexOperationQueueName = context.iexProxy().
                        xIndexOperationQueueNameUpdate();
                }
                if (xIndexOperationQueueName != null) {
                    generator.addHeader(
                        YandexHeaders.X_INDEX_OPERATION_QUEUE,
                        xIndexOperationQueueName);
                }
                Header zooShardId = session.request().getFirstHeader(
                    YandexHeaders.ZOO_SHARD_ID);
                if (zooShardId != null) {
                    generator.addHeader(zooShardId);
                }
                AsyncClient producerAsyncClient =
                    context.iexProxy().producerAsyncClient().adjust(
                        session.context());
                session.logger().info("Delete no_facts docs");
                producerAsyncClient.execute(
                    config.producerAsyncClientConfig().host(),
                    generator,
                    AsyncStringConsumerFactory.OK,
                    session.listener()
                        .createContextGeneratorFor(producerAsyncClient),
                    new DeleteNoFactsCallback(context));
            } catch (IOException | BadRequestException
                | JsonUnexpectedTokenException e)
            {
                session.logger().log(
                    Level.SEVERE,
                    "Delete request url and body building error",
                    e);
                session.handleException(
                    HttpExceptionConverter.toHttpException(e));
            }
        }

        private String getUrl()
            throws BadRequestException, JsonUnexpectedTokenException
        {
            QueryConstructor query = new QueryConstructor("/delete?facts");
            String queueName = context.iexProxy().config()
                .factsIndexingQueueName();
            query.append("service", queueName);
            Long operationId = ValueUtils.asLongOrNull(
                context.json().get(PgFields.OPERATION_ID));
            if (operationId != null) {
                query.append("operationId", operationId);
            }
            return query.toString();
        }

        private String getBody(final List<Long> mids) throws IOException {
            StringBuilderWriter sbw = new StringBuilderWriter();
            try (JsonWriter writer = new JsonWriter(sbw)) {
                writer.startObject();
                writer.key("prefix");
                writer.value(context.prefix());
                writer.key("docs");
                writer.startArray();
                for (Long mid : mids) {
                    writer.startObject();
                    String url = "facts_" + context.prefix() + '_' + mid
                        + '_' + NO_FACTS;
                    writer.key("url");
                    writer.value(url);
                    writer.endObject();
                }
                writer.endArray();
                writer.endObject();
            }
            return sbw.toString();
        }
    }

    private static class DeleteNoFactsCallback
        extends AbstractProxySessionCallback<Object>
    {
        private ChangeContext context;

        DeleteNoFactsCallback(final ChangeContext context) {
            super(context.session());
            this.context = context;
        }

        @Override
        public void completed(final Object response) {
            try {
                UpdateHandler.INSTANCE.handle(context);
            } catch (BadRequestException | JsonUnexpectedTokenException e) {
                session.logger().log(
                    Level.SEVERE,
                    "UpdateHandler execution for messages moved from spam"
                    + " failed",
                    e);
                session.response(HttpStatus.SC_BAD_REQUEST);
            }
        }
    }

    private static class RemoveTrustedUserCallback
        extends AbstractFilterFutureCallback<String, Void>
    {
        private final AbstractCallback<Void> callback;

        RemoveTrustedUserCallback(final AbstractCallback<Void> callback) {
            super(callback);
            this.callback = callback;
        }

        @Override
        public void completed(final String result) {
            callback.context().abstractContext().session().logger().info(
                "User remove request is succesfully sent: "
                + result);
            callback.completed(null);
        }
    }

    private static class CompainsActionsFilterSearchHandler
        extends AbstractFilterSearchCallback<Void>
    {
        private final Set<String> falseMids;
        private final Set<String> spamSamplesMids;

        CompainsActionsFilterSearchHandler(
            final ChangeContext context,
            final FutureCallback<List<Void>> callback,
            final Set<String> falseMids,
            final Set<String> spamSamplesMids)
            throws BadRequestException
        {
            super(
                context,
                callback,
                Sets.union(falseMids, spamSamplesMids));
            this.falseMids = falseMids;
            this.spamSamplesMids = spamSamplesMids;
        }

        public boolean skipEmptyEntities() {
            return false;
        }

        public boolean skipSpam() {
            return false;
        }

        public AbstractCallback<Void> subMessageCallback(
            final IndexationContext<Void> context)
        {
            return new CompainsActionFinishedCallback(context);
        }

        private void falseComplaint(final AbstractCallback<Void> callback) {
            //get Trusted user uid from email
            final IndexationContext<Void> msgContext = callback.context();

            UserActionHandler.requestTikaite(
                msgContext.abstractContext(),
                "/headers",
                List.of("stid=" + msgContext.stid()),
                new TikaiteHeadersCallback(callback));
        }

        private void spamSample(final AbstractCallback<Void> callback) {
            final IndexationContext<Void> msgContext = callback.context();

            UserActionHandler.requestTikaite(
                msgContext.abstractContext(),
                "/extract",
                List.of("extractor-name=nested-mail", "stid=" + msgContext.stid()),
                new TikaiteDocsCallback(callback));
        }

        public void executeSubCallback(final AbstractCallback<Void> callback) {
            String mid = callback.context().mid();
            if (falseMids.contains(mid)) {
                falseComplaint(callback);
            } else if (spamSamplesMids.contains(mid)) {
                spamSample(callback);
            } else {
                callback.context().abstractContext().session().logger()
                    .warning("Unexpected mid: " + mid);
                callback.completed(null);
            }
        }
    }

    private static class CompainsActionFinishedCallback
        extends AbstractCallback<Void>
    {
        CompainsActionFinishedCallback(final IndexationContext<Void> context) {
            super(context);
        }

        @Override
        public void completed(final Void result) {
            context.abstractContext().session().logger().info(
                "Compains actions finished for: "
                    + "mid: " + context.mid()
                    + ", subj: " + context.subject());
            context.callback().completed(result);
        }
    }

    private static class TikaiteHeadersCallback
        extends AbstractFilterFutureCallback<JsonObject, Void>
    {
        private final AbstractCallback<Void> callback;

        TikaiteHeadersCallback(final AbstractCallback<Void> callback) {
            super(callback);
            this.callback = callback;
        }

        @Override
        public void completed(final JsonObject result) {
            try {
                final IndexationContext<Void> msgContext = callback.context();
                final ChangeContext context =
                    (ChangeContext) msgContext.abstractContext();
                String complainerUid = null;
                String to = null;
                JsonMap map = result.asMap();
                JsonList headers = map.getList("headers");
                for (JsonObject o: headers) {
                    JsonMap header = o.asMap();
                    if (header.size() == 0) {
                        continue;
                    }
                    Map.Entry<String, JsonObject> headerKV =
                        header.entrySet().iterator().next();
                    String key = headerKV.getKey();
                    context.session().logger().info(
                        "Iterating header: " + key);
                    if (key.equalsIgnoreCase("x-yandex-complainer-uid")) {
                        complainerUid = headerKV.getValue().asString();
                    } else if (key.equalsIgnoreCase("to")) {
                        to = headerKV.getValue().asString();
                    }
                }
                context.session().logger().info("Tikaite headers: "
                    + JsonType.NORMAL.toString(headers));
                Long uid = null;
                if (complainerUid != null) {
                    context.session().logger().info(
                        "Found complainerUid header: " + complainerUid);
                    uid = Long.parseLong(complainerUid);
                } else if (to != null) {
                    context.session().logger().info(
                        "Found <To> header: " + to);
                    String pattern = "^.*?([0-9]+)@uid.ya$";
                    Pattern r = Pattern.compile(pattern);
                    Matcher m = r.matcher(to);
                    if (m.find()) {
                        uid = Long.parseLong(m.group(1));
                    }
                }
                if (uid == null) {
                    context.session().logger().warning(
                        "Can't find appropriate complainer uid for mid: "
                            + msgContext.mid()
                            + ", subj: " + msgContext.subject());
                    callback.completed(null);
                } else {
                    removeTrustedUser(uid);
                }
            } catch (HttpException | JsonException e) {
                failed(e);
            }
        }

        private void removeTrustedUser(final long uid) throws HttpException {
            final IndexationContext<Void> msgContext = callback.context();
            final ChangeContext context =
                (ChangeContext) msgContext.abstractContext();
            final IexProxy iexProxy = context.iexProxy();
            String queueName = iexProxy.
                config().factsIndexingQueueName();
            HttpHost host =
                iexProxy.config().producerAsyncClientConfig()
                    .host();

            String trustedKey = trustedKey(uid);
            QueryConstructor query = new QueryConstructor(
                "/delete?so_trusted_user");
            query.append("prefix", uid);
            query.append("service", queueName);
            query.append("url", trustedKey);
            StringBuilderWriter sbw = new StringBuilderWriter();
            try (JsonWriter writer = new JsonWriter(sbw)) {
                writer.startObject();
                writer.key("prefix");
                writer.value(uid);
                writer.key("docs");
                writer.startArray();
                writer.startObject();
                writer.key("url");
                writer.value(trustedKey);
                writer.endObject();
                writer.endArray();
                writer.endObject();
            } catch (IOException e) {
                failed(e);
            }
            context.session().logger().info(
                "Removing trusted user: " + trustedKey);
            BasicAsyncRequestProducerGenerator generator =
                new BasicAsyncRequestProducerGenerator(
                    query.toString(),
                    sbw.toString());
            generator.addHeader(YandexHeaders.SERVICE, queueName);
            AsyncClient producerAsyncClient =
                iexProxy.producerAsyncClient().adjust(
                    context.session().context());
            producerAsyncClient.execute(
                host,
                generator,
                AsyncStringConsumerFactory.OK,
                context.session().listener()
                    .createContextGeneratorFor(producerAsyncClient),
                new RemoveTrustedUserCallback(callback));
        }
    }

    private static class TikaiteDocsCallback
        extends AbstractFilterFutureCallback<JsonObject, Void>
    {
        private final AbstractCallback<Void> callback;

        TikaiteDocsCallback(final AbstractCallback<Void> callback) {
            super(callback);
            this.callback = callback;
        }

        @Override
        public void completed(final JsonObject result) {
            try {
                final IndexationContext<Void> msgContext = callback.context();
                final ChangeContext context =
                    (ChangeContext) msgContext.abstractContext();
                final IexProxy iexProxy = context.iexProxy();
                String url = spamSampleUrl(context, msgContext.mid());
                Object operationId = context.json().get(PgFields.OPERATION_ID);
                StringBuilderWriter sbw = new StringBuilderWriter();
                try (JsonWriter writer = JsonType.DOLLAR.create(sbw)) {
                    writer.startObject();
                    writer.key("prefix");
                    writer.value(context.prefix());
                    writer.key("docs");
                    writer.startArray();
                    writer.startObject();
                    writer.key("url");
                    writer.value(url);
                    writer.key("spam_sample_type");
                    writer.value("so_compains");
                    writer.key("spam_sample_revision");
                    writer.value(operationId);
                    String labels =
                        msgContext.meta().get(MailIndexFields.LABELS_NAMES);
                    if (labels != null) {
                        writer.key("spam_sample_labels");
                        // Yandex.Mail replaces all spaces in label names with
                        // nbsp
                        writer.value('\n' + labels.replace('\u00a0', ' '));
                    }
                    writer.key("spam_sample_stid");
                    writer.value(msgContext.stid());
                    writer.key("spam_sample_data");
                    writer.value(JsonType.NORMAL.toString(result));
                    writer.endObject();
                    writer.endArray();
                    writer.endObject();
                }
                BasicAsyncRequestProducerGenerator generator =
                    new BasicAsyncRequestProducerGenerator(
                        "/modify?spam-samples&prefix=" + context.prefix()
                        + "&url=" + url
                        + "&operation-id=" + operationId,
                        sbw.toString());
                generator.addHeader(YandexHeaders.SERVICE, context.mailSearchQueueName());

                AsyncClient producerAsyncClient =
                    context.iexProxy().producerAsyncClient().adjust(
                        context.session().context());

                producerAsyncClient.execute(
                    iexProxy.config().producerAsyncClientConfig().host(),
                    generator,
                    AsyncStringConsumerFactory.OK,
                    context.session().listener()
                        .createContextGeneratorFor(producerAsyncClient),
                    new AbstractFilterFutureCallback<String, Void>(callback) {
                        @Override
                        public void completed(String s) {
                            callback.completed(null);
                        }
                    });
            } catch (IOException e) {
                failed(e);
            }
        }
    }

    private static class CompainsHookCallback
        extends AbstractProxySessionCallback<Map.Entry<List<Void>, Object>>
    {
        private final ChangeContext context;

        CompainsHookCallback(final ChangeContext context) {
            super(context.session());
            this.context = context;
        }

        @Override
        public void completed(final Map.Entry<List<Void>, Object> result) {
            context.session().logger().info("All compains actions are taken");
            context.session().response(HttpStatus.SC_OK);
        }
    }
}
