package ru.yandex.iex.proxy.move;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.logging.Level;

import org.apache.http.concurrent.FutureCallback;

import ru.yandex.dbfields.PgFields;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.nio.AsyncStringConsumerFactory;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.iex.proxy.ChangeContext;
import ru.yandex.iex.proxy.IndexationContext;
import ru.yandex.iex.proxy.complaints.MailMessageContext;
import ru.yandex.iex.proxy.complaints.UserAction;
import ru.yandex.iex.proxy.complaints.UserActionHandler;
import ru.yandex.iex.proxy.xutils.spamreport.SpamReportSender;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonNull;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.parser.mail.received.ReceivedChainParser;
import ru.yandex.parser.mail.senders.SendersContext;
import ru.yandex.parser.uri.QueryConstructor;

public class SingleMessageHeadersCallback
    extends AbstractFilterFutureCallback<JsonObject, UpdateDataHolder>
{
    private static final String RECEIVED = "received";
    private static final String X_YANDEX_SPAM = "x-yandex-spam";
    private static final double MAX_FRESH_AGE = 21600d;
    private static final double MAX_REPORT_AGE = 48 * 3600d;
    private static final int MIN_NEURO_HARD_WORDS = 6;

    private final IndexationContext<UpdateDataHolder> indexationContext;
    private final ChangeContext context;
    private final UserAction action;
    private final Map<?, ?> actionInfo;
    private final Boolean sendReports;

    public SingleMessageHeadersCallback(
        final FutureCallback<UpdateDataHolder> callback,
        final ChangeContext context,
        final IndexationContext<UpdateDataHolder> indexationContext,
        final UserAction action,
        final Map<?, ?> actionInfo,
        final Boolean sendReports)
    {
        super(callback);
        this.indexationContext = indexationContext;
        this.context = context;
        this.action = action;
        this.actionInfo = actionInfo;
        this.sendReports = sendReports;
    }

    @Override
    public void completed(final JsonObject result) {
        if (action == null || action == UserAction.OTHER) {
            callback.completed(null);
            return;
        }
        UpdateDataHolder updateDoc;
        try {
            updateDoc = processTikaiteAnswer(result);
        } catch (JsonException s) {
            failed(s);
            return;
        }
        finalProcessing(updateDoc);
    }

    protected UpdateDataHolder processTikaiteAnswer(final JsonObject result) throws JsonException {
        Map<String, List<String>> headersMap = new HashMap<>();
        StringBuilder sb = new StringBuilder();
        sb.append('\n');
        ReceivedChainParser chainParser = new ReceivedChainParser(context.iexProxy().yandexNets());
        JsonList headers = result.asMap().getList("headers");
        for (JsonObject header : headers) {
            Map.Entry<String, JsonObject> headerKV = header.asMap().entrySet().iterator().next();
            String key = headerKV.getKey().toLowerCase(Locale.ROOT);
            String value = headerKV.getValue().asString();
            sb.append(key);
            sb.append(':');
            sb.append(' ');
            sb.append(value);
            sb.append('\n');
            headersMap.computeIfAbsent(key, x -> new ArrayList<>()).add(value);
            if (key.equals(RECEIVED)) {
                chainParser.process(value);
            }
        }
        SendersContext sendersContext = new SendersContext(new String(sb));
        return
            new UpdateDataHolder(
                new MailMessageContext(context, action, Math.round(context.operationDate()), chainParser)
                    .headersMap(headersMap).indexationContext(indexationContext).actionInfo(actionInfo),
                chainParser.sourceDomain(),
                sendersContext.extractSenders());
    }

    protected void finalProcessing(final UpdateDataHolder updateDoc) {
        if (action == UserAction.SPAM) {
            spamHook(updateDoc, callback);
        } else {
            callback.completed(updateDoc);
        }
    }

    private void spamHook(
        final UpdateDataHolder updateDoc,
        final FutureCallback<? super UpdateDataHolder> callback)
    {
        double age = context.operationDate() - indexationContext.receivedDate();
        if (sendReports != null && sendReports && age < MAX_REPORT_AGE) {
            if (age < MAX_FRESH_AGE) {
                context.iexProxy().trustedSpamFreshComplaint();
            }
            UserActionHandler.requestTikaite(
                context,
                "/extract",
                List.of("stid=" + indexationContext.stid()),
                new TikaiteExtractCallback(
                    callback,
                    indexationContext,
                    updateDoc));
        } else {
            callback.completed(updateDoc);
        }
    }

    private static class TikaiteExtractCallback
        extends AbstractFilterFutureCallback<JsonObject, UpdateDataHolder>
    {
        private final IndexationContext<UpdateDataHolder> indexationContext;
        private final UpdateDataHolder updateDoc;

        TikaiteExtractCallback(
            final FutureCallback<? super UpdateDataHolder> callback,
            final IndexationContext<UpdateDataHolder> indexationContext,
            final UpdateDataHolder updateDoc)
        {
            super(callback);
            this.indexationContext = indexationContext;
            this.updateDoc = updateDoc;
        }

        @Override
        public void completed(final JsonObject result) {
            try {
                List<NeuroHardInput> inputs = new ArrayList<>();
                for (JsonObject doc: result.get("docs").asList()) {
                    String hid = doc.get("hid").asString();
                    Long wordCount = doc.get("word_count").asLongOrNull();
                    if (wordCount != null
                        && wordCount >= MIN_NEURO_HARD_WORDS)
                    {
                        JsonObject embedding = doc.get("fast_text_embedding");
                        if (embedding != JsonNull.INSTANCE) {
                            inputs.add(
                                new NeuroHardInput(
                                    hid,
                                    wordCount,
                                    embedding.get("embedding").asList()));
                        }
                    }
                }
                updateDoc.context().spamSampleData(
                    JsonType.NORMAL.toString(result));
                NeuroHardCallback neuroHardCallback =
                    new NeuroHardCallback(
                        callback,
                        indexationContext,
                        updateDoc,
                        inputs);
                if (inputs.isEmpty()) {
                    neuroHardCallback.completed(Collections.emptyList());
                } else {
                    ChangeContext context =
                        (ChangeContext) indexationContext.abstractContext();
                    AsyncClient client =
                        context.iexProxy().neuroHardsClient()
                            .adjust(context.session().context());
                    MultiFutureCallback<JsonObject> multiCallback =
                        new MultiFutureCallback<>(neuroHardCallback);
                    for (NeuroHardInput input: inputs) {
                        QueryConstructor query = new QueryConstructor(
                            "/score?b=-1.5&f=1.8&r=-0.17&t=-0.2&eps=0.15");
                        query.append("hid", input.hid);
                        query.append("word-count", input.wordCount);
                        query.append(
                            "embed",
                            JsonType.NORMAL.toString(input.embedding));
                        client.execute(
                            context.iexProxy().neuroHardsHost(),
                            new BasicAsyncRequestProducerGenerator(
                                query.toString()),
                            JsonAsyncTypesafeDomConsumerFactory.OK,
                            context.session().listener()
                                .createContextGeneratorFor(client),
                            multiCallback.newCallback());
                    }
                    multiCallback.done();
                }
            } catch (JsonException | BadRequestException e) {
                failed(e);
            }
        }
    }

    private static class NeuroHardCallback
        extends AbstractFilterFutureCallback
            <List<JsonObject>, UpdateDataHolder>
    {
        private final IndexationContext<UpdateDataHolder> context;
        private final UpdateDataHolder updateDoc;
        private final List<NeuroHardInput> inputs;

        NeuroHardCallback(
            final FutureCallback<? super UpdateDataHolder> callback,
            final IndexationContext<UpdateDataHolder> context,
            final UpdateDataHolder updateDoc,
            final List<NeuroHardInput> inputs)
        {
            super(callback);
            this.context = context;
            this.updateDoc = updateDoc;
            this.inputs = inputs;
        }

        @Override
        public void completed(final List<JsonObject> results) {
            NeuroHardStoreCallback callback = new NeuroHardStoreCallback(
                super.callback,
                updateDoc,
                context,
                results);
            int size = results.size();
            List<NeuroHardResult> hids = new ArrayList<>(size);
            for (int i = 0; i < size; ++i) {
                JsonObject result = results.get(i);
                if (result.type() == JsonObject.Type.MAP) {
                    try {
                        JsonMap data = result.asMap();
                        JsonObject score = data.get("score");
                        if (score != JsonNull.INSTANCE
                            && score.asDouble() >= 2.95)
                        {
                            hids.add(
                                new NeuroHardResult(
                                    inputs.get(i).hid,
                                    data.get("dist").asDouble()));
                        }
                    } catch (JsonException e) {
                        context.abstractContext().session().logger().log(
                            Level.WARNING,
                            "Failed to parse neurohard for hid "
                            + inputs.get(i).hid + ':' + ' '
                            + JsonType.NORMAL.toString(result),
                            e);
                    }
                }
            }
            if (hids.isEmpty()) {
                callback.completed(null);
            } else {
                ChangeContext context =
                    (ChangeContext) this.context.abstractContext();
                String sampleUrl =
                    "spam_samples_" + context.prefix()
                    + "_neuro_hard_" + this.context.mid();
                // TODO: something more unique
                Object operationId = context.json().get(PgFields.OPERATION_ID);
                if (operationId == null) {
                    callback.completed(null);
                    return;
                }
                StringBuilderWriter sbw = new StringBuilderWriter();
                try (JsonWriter writer = JsonType.DOLLAR.create(sbw)) {
                    writer.startObject();
                    writer.key("prefix");
                    writer.value(MoveHandler.COMPAINS_UID);
                    writer.key("docs");
                    writer.startArray();
                    for (NeuroHardResult result: hids) {
                        writer.startObject();
                        writer.key("url");
                        writer.value(sampleUrl + '_' + result.hid);
                        writer.key("spam_sample_type");
                        writer.value("neuro_hard");
                        writer.key("spam_sample_revision");
                        writer.value(operationId);
                        writer.key("spam_sample_labels");
                        writer.value(
                            "neuro_dist_" + result.distance
                            + "\nhid_" + result.hid);
                        writer.key("spam_sample_stid");
                        writer.value(this.context.stid());
                        writer.key("spam_sample_data");
                        writer.value(updateDoc.context().spamSampleData());
                        writer.endObject();
                    }
                    writer.endArray();
                    writer.endObject();
                } catch (IOException e) {
                    callback.failed(e);
                    return;
                }
                BasicAsyncRequestProducerGenerator generator =
                    new BasicAsyncRequestProducerGenerator(
                        "/modify?spam-samples&prefix="
                        + MoveHandler.COMPAINS_UID
                        + "&neuro-hard&url=" + sampleUrl,
                        sbw.toString());
                generator.addHeader(YandexHeaders.SERVICE, context.mailSearchQueueName());

                AsyncClient producerAsyncClient =
                    context.iexProxy().producerAsyncClient().adjust(
                        context.session().context());

                producerAsyncClient.execute(
                    context.iexProxy().config().producerAsyncClientConfig()
                        .host(),
                    generator,
                    AsyncStringConsumerFactory.OK,
                    context.session().listener()
                        .createContextGeneratorFor(producerAsyncClient),
                    callback);
            }
        }

        @Override
        public void failed(final Exception e) {
            context.abstractContext().session().logger().log(
                Level.WARNING,
                "Neuro hards failed",
                e);
            completed(Collections.emptyList());
        }
    }

    private static class NeuroHardStoreCallback
        extends AbstractFilterFutureCallback<String, UpdateDataHolder>
    {
        private final UpdateDataHolder updateDoc;
        private final IndexationContext<UpdateDataHolder> context;
        private final List<JsonObject> neuroHardResult;

        NeuroHardStoreCallback(
            final FutureCallback<? super UpdateDataHolder> callback,
            final UpdateDataHolder updateDoc,
            final IndexationContext<UpdateDataHolder> context,
            final List<JsonObject> neuroHardResult)
        {
            super(callback);
            this.updateDoc = updateDoc;
            this.context = context;
            this.neuroHardResult = neuroHardResult;
        }

        @Override
        public void completed(final String result) {
            List<String> xYandexSpamList =
                updateDoc.context().headersMap().get(X_YANDEX_SPAM);
            String xYandexSpam = null;
            if (xYandexSpamList != null && !xYandexSpamList.isEmpty()) {
                xYandexSpam = xYandexSpamList.get(0);
            }
            SpamReportSender.send(
                new SpamContext(context),
                new SpamHookCallback(context, callback, updateDoc),
                "4".equals(xYandexSpam),
                neuroHardResult);
        }
    }

    private static class SpamHookCallback
        extends AbstractFilterFutureCallback<Void, UpdateDataHolder>
    {
        private final UpdateDataHolder updateDoc;
        private final IndexationContext<UpdateDataHolder> context;

        public SpamHookCallback(
            IndexationContext<UpdateDataHolder> context,
            FutureCallback<? super UpdateDataHolder> callback,
            UpdateDataHolder updateDoc)
        {
            super(callback);
            this.updateDoc = updateDoc;
            this.context = context;
        }

        @Override
        public void completed(final Void result) {
            context.abstractContext().session().logger().info(
                "Spam report sent for: mid: " + context.mid() + ", subj: " + context.subject());
            callback.completed(updateDoc);
        }
    }

    private static class NeuroHardInput {
        private final String hid;
        private final long wordCount;
        private final JsonList embedding;

        NeuroHardInput(
            final String hid,
            final long wordCount,
            final JsonList embedding)
        {
            this.hid = hid;
            this.wordCount = wordCount;
            this.embedding = embedding;
        }
    }

    private static class NeuroHardResult {
        private final String hid;
        private final double distance;

        NeuroHardResult(final String hid, final double distance) {
            this.hid = hid;
            this.distance = distance;
        }
    }
}
