package ru.yandex.iex.proxy.move;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.logging.Level;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.apache.http.HttpHost;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.client.so.shingler.AbstractShinglesMap;
import ru.yandex.client.so.shingler.ActivityShinglerClient;
import ru.yandex.client.so.shingler.ActivityShingles;
import ru.yandex.client.so.shingler.ComplShinglerClient;
import ru.yandex.client.so.shingler.ComplShingles;
import ru.yandex.client.so.shingler.FreemailShinglerClient;
import ru.yandex.client.so.shingler.FreemailShingles;
import ru.yandex.client.so.shingler.MassShinglerClient;
import ru.yandex.client.so.shingler.MassShinglerResult;
import ru.yandex.client.so.shingler.SenderShinglerClient;
import ru.yandex.client.so.shingler.SenderShingles;
import ru.yandex.client.so.shingler.ShingleException;
import ru.yandex.client.so.shingler.UrlShinglerClient;
import ru.yandex.client.so.shingler.UrlShingles;
import ru.yandex.client.so.shingler.config.ShinglerType;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.nio.AsyncStringConsumerFactory;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.iex.proxy.complaints.AbstractUserActionContext;
import ru.yandex.iex.proxy.complaints.ImmutableComplaintsConfig;
import ru.yandex.json.writer.JsonType;
import ru.yandex.parser.mail.senders.SenderInfo;

public enum UpdateDataExecutor {
    PFILTERS {
        public Object getKey(final UpdateDataHolder data) {
            return new Object[]{data.senderDomain(), data.senders(), data.isSpam()};
        }

        public UpdateDataHolder reduce(UpdateDataHolder base, final UpdateDataHolder other) {
            if (base.receivedDate() < other.receivedDate()) {
                base.setReceivedDate(other.receivedDate());
            }
            return base;
        }

        // Almost all logic for personal filter's incrementing contained here
        public Stream<Map<String, Object>> generateLuceneDoc(UpdateDataHolder data) {
            int sendersSize = data.senders().size();
            if (sendersSize == 0) {
                return Stream.empty();
            }
            String relevantCounter;
            String type;
            String antitype;
            if (data.isSpam()) {
                relevantCounter = "pfilters_spams";
                type = "spam";
                antitype = "ham";
            } else {
                relevantCounter = "pfilters_hams";
                type = "ham";
                antitype = "spam";
            }
            List<Map<String, Object>> docs = new ArrayList<>(sendersSize);
            for (SenderInfo senderInfo: data.senders()) {
                docs.add(
                    Map.of(
                        "url",
                        senderInfo.type().pfiltersPrefix() + data.uid() + '_'
                            + senderInfo.email() + '/' + data.senderDomain(),
                        "pfilters_last_timestamp",
                        data.operationDate(),
                        "pfilters_last_type",
                        mkFunc(
                            "if",
                            mkFunc(
                                "eq",
                                antitype,
                                mkFunc(
                                    "get",
                                    "pfilters_last_type")),
                            null,
                            type),
                        relevantCounter,
                        mkFunc("inc")));
            }
            return docs.stream();
        }

        public List<Map<String, Object>> generateDocs(final List<UpdateDataHolder> lists) {
            return lists.stream()
                    .collect(Collectors.groupingBy(this::getKey,
                            Collectors.reducing(this::reduce)))
                    .values().stream().filter(Optional::isPresent).map(Optional::get)
                    .flatMap(this::generateLuceneDoc)
                    .collect(Collectors.toList());
        }

        @Override
        public void updateCounters(
            final AbstractUserActionContext context,
            final List<UpdateDataHolder> lists,
            final FutureCallback<String> callback,
            final String addonParams)
        {
            updateDocs(context, generateDocs(lists), callback, addonParams);
        }
    },
    ABUSES {
        public Object getKey(final UpdateDataHolder data) {
            return new Object[]{data.stid(), data.isSpam()};
        }

        public UpdateDataHolder reduce(UpdateDataHolder base, final UpdateDataHolder other) {
            if (base.operationDate() < other.operationDate()) {
                base.setOperationDate(other.operationDate());
            }
            return base;
        }

        // Almost all logic for complaint's incrementing contained here
        public Stream<Map<String, Object>> generateLuceneDoc(UpdateDataHolder data) {
            if (data.stid() == null) {
                return Stream.empty();
            }
            Map<String, ?> updateCondition1 = mkFunc("gt",
                mkFunc("get", "abuses_first_timestamp"), data.operationDate());
            Map<String, ?> updateCondition2 = mkFunc("lt",
                mkFunc("get", "abuses_last_timestamp"), data.operationDate());
            String abuseType = data.isSpam() ? "spam" : "ham";

            return Stream.of(
                Map.of(
                    "url", abusesUrl(data.uid(), data.stid(), abuseType),
                    "abuses_type", abuseType,
                    "abuses_uid", Long.parseLong(data.uid()),
                    "abuses_source", data.source().name(),
                    "abuses_msg_date", data.receivedDate(),
                    "abuses_cmpl_date", data.operationDate(),
                    "abuses_cnt", mkFunc("inc", 1),
                    "abuses_smtp_id", data.queueId() == null ? "" : data.queueId(),
                    "abuses_first_timestamp", mkFunc(
                        "set", data.operationDate(), mkFunc("or",
                           mkFunc("eq",mkFunc("get", "abuses_first_timestamp"), "0"),
                           updateCondition1)),
                    "abuses_last_timestamp", mkFunc("set", data.operationDate(), updateCondition2)));
        }

        public List<Map<String, Object>> generateDocs(final List<UpdateDataHolder> lists) {
            return lists.stream()
                    .filter(x -> !x.isRetry())
                    .collect(Collectors.groupingBy(this::getKey,
                        Collectors.reducing(this::reduce)))
                    .values().stream().filter(Optional::isPresent).map(Optional::get)
                    .flatMap(this::generateLuceneDoc)
                    .collect(Collectors.toList());
        }

        @Override
        public void updateCounters(
            final AbstractUserActionContext context,
            final List<UpdateDataHolder> lists,
            final FutureCallback<String> callback,
            final String addonParams)
        {
            updateDocs(context, generateDocs(lists), callback, addonParams);
        }
    },
    ACTIVITY_SHINGLER {
        @Override
        public void updateCounters(
            final AbstractUserActionContext context,
            final List<UpdateDataHolder> lists,
            final FutureCallback<String> callback,
            final String addonParams)
        {
            final ActivityShingles shingles = new ActivityShingles();
            try {
                aggregateShinglerData(context, lists, shingles, ShinglerType.ACTIVITY);
            } catch (Exception e) {
                context.session().logger().log(
                    Level.SEVERE,
                    "Failed to aggregate shingler's data for Activity-shingler's request",
                    e);
                callback.completed(null);
                return;
            }
            if (shingles.size() > 0) {
                ImmutableComplaintsConfig complaintsConfig = context.iexProxy().config().complaintsConfig();
                ActivityShinglerClient client =
                    (ActivityShinglerClient) complaintsConfig.shinglerClient(ShinglerType.ACTIVITY)
                        .adjust(context.session().context());
                if (client == null) {
                    context.session().logger().log(
                        Level.SEVERE,
                        "UpdateDataExecutor.ACTIVITY_SHINGLER.updateCounters failed to create client!");
                    callback.completed(null);
                } else {
                    try {
                        final String putBody = client.getPutQuery(shingles);
                        context.session().logger().info("Activity-shingler's putBody: " + putBody);
                        if (putBody == null || putBody.isEmpty()) {
                            context.session().logger().log(
                                Level.SEVERE,
                                "Failed to generate PUT query from Activity-shingler's data");
                            callback.completed(null);
                        } else if (complaintsConfig.shinglersDryRun()) {
                            context.session().logger().info(
                                "Request to Activity-shingler is omitted due to dry-run mode switched on");
                            callback.completed(null);
                        } else {
                            if (complaintsConfig.shinglersAddStatByActions()
                                && complaintsConfig.shinglersByUserActionStaters().containsKey(ShinglerType.ACTIVITY))
                            {
                                client = (ActivityShinglerClient) client
                                    .adjustStater(
                                        complaintsConfig.shinglersByUserActionStaters()
                                            .get(ShinglerType.ACTIVITY)
                                            .get(context.action()));
                            }
                            client.putShingles(putBody, callback);
                        }
                    } catch (ShingleException e) {
                        context.session().logger().log(
                            Level.SEVERE,
                            "Failed to generate PUT query from Activity-shingler's data",
                            e);
                        callback.failed(e);
                    }
                }
            } else {
                callback.completed(null);
            }
        }
    },
    COMPL_SHINGLER {
        @Override
        public void updateCounters(
            final AbstractUserActionContext context,
            final List<UpdateDataHolder> lists,
            final FutureCallback<String> callback,
            final String addonParams)
        {
            final ComplShingles shingles = new ComplShingles();
            try {
                aggregateShinglerData(context, lists, shingles, ShinglerType.COMPL);
                //context.session().logger().info("UpdateDataExecutor.COMPL_SHINGLER.updateCounters: shingles="
                //    + shingles);
            } catch (Exception e) {
                context.session().logger().log(
                    Level.SEVERE,
                    "Failed to aggregate shingler's data for Compl-shingler's request",
                    e);
                callback.completed(null);
                return;
            }
            if (shingles.size() > 0) {
                ImmutableComplaintsConfig complaintsConfig = context.iexProxy().config().complaintsConfig();
                ComplShinglerClient client =
                    (ComplShinglerClient) complaintsConfig.shinglerClient(ShinglerType.COMPL)
                        .adjust(context.session().context());
                if (client == null) {
                    context.session().logger().log(
                        Level.SEVERE,
                        "UpdateDataExecutor.COMPL_SHINGLER.updateCounters failed to create client!");
                    callback.completed(null);
                } else {
                    try {
                        final String putBody = client.getPutQuery(shingles);
                        context.session().logger().info("Compl-shingler's putBody: " + putBody);
                        if (putBody == null || putBody.isEmpty()) {
                            context.session().logger().log(
                                Level.SEVERE,
                                "Failed to generate PUT query from Compl-shingler's data");
                            callback.completed(null);
                        } else if (complaintsConfig.shinglersDryRun()) {
                            context.session().logger().info(
                                "Request to Compl-shingler is omitted due to dry-run mode switched on");
                            callback.completed(null);
                        } else {
                            if (complaintsConfig.shinglersAddStatByActions()
                                && complaintsConfig.shinglersByUserActionStaters().containsKey(ShinglerType.COMPL))
                            {
                                client = (ComplShinglerClient) client
                                    .adjustStater(
                                        complaintsConfig.shinglersByUserActionStaters()
                                            .get(ShinglerType.COMPL)
                                            .get(context.action()));
                            }
                            client.putShingles(putBody, callback);
                        }
                    } catch (ShingleException e) {
                        context.session().logger().log(
                            Level.SEVERE,
                            "Failed to generate PUT query from Compl-shingler's data",
                            e);
                        callback.failed(e);
                    }
                }
            } else {
                callback.completed(null);
            }
        }
    },
    FREEMAIL_SHINGLER {
        @Override
        public void updateCounters(
            final AbstractUserActionContext context,
            final List<UpdateDataHolder> lists,
            final FutureCallback<String> callback,
            final String addonParams)
        {
            final FreemailShingles shingles = new FreemailShingles();
            try {
                aggregateShinglerData(context, lists, shingles, ShinglerType.FREEMAIL);
            } catch (Exception e) {
                context.session().logger().log(
                    Level.SEVERE,
                    "Failed to aggregate shingler's data for Freemail-shingler's request",
                    e);
                callback.completed(null);
                return;
            }
            if (shingles.size() > 0) {
                ImmutableComplaintsConfig complaintsConfig = context.iexProxy().config().complaintsConfig();
                FreemailShinglerClient client =
                    (FreemailShinglerClient) complaintsConfig.shinglerClient(ShinglerType.FREEMAIL)
                        .adjust(context.session().context());
                if (client == null) {
                    context.session().logger().log(
                        Level.SEVERE,
                        "UpdateDataExecutor.FREEMAIL_SHINGLER.updateCounters failed to create client!");
                    callback.completed(null);
                } else {
                    try {
                        String putBody = client.getPutQuery(shingles, false);
                        context.session().logger().info("Freemail-shingler's putBody: " + putBody);
                        if (putBody == null || putBody.isEmpty()) {
                            context.session().logger().log(
                                Level.SEVERE,
                                "Failed to generate PUT query from Freemail-shingler's data");
                            callback.completed(null);
                        } else if (complaintsConfig.shinglersDryRun()) {
                            context.session().logger().info(
                                "Request to Freemail-shingler is omitted due to dry-run mode switched on");
                            callback.completed(null);
                        } else {
                            if (complaintsConfig.shinglersAddStatByActions()
                                && complaintsConfig.shinglersByUserActionStaters().containsKey(ShinglerType.FREEMAIL))
                            {
                                client = (FreemailShinglerClient) client
                                    .adjustStater(
                                        complaintsConfig.shinglersByUserActionStaters()
                                            .get(ShinglerType.FREEMAIL)
                                            .get(context.action()));
                            }
                            client.putShingles(putBody, callback);
                        }
                    } catch (ShingleException e) {
                        context.session().logger().log(
                            Level.SEVERE,
                            "Failed to generate PUT query from Freemail-shingler's data",
                            e);
                        callback.failed(e);
                    }
                }
            } else {
                callback.completed(null);
            }
        }
    },
    MASS_IN_SHINGLER {
        @Override
        public void updateCounters(
            final AbstractUserActionContext context,
            final List<UpdateDataHolder> lists,
            final FutureCallback<String> callback,
            final String addonParams)
        {
            final MassShinglerResult shingles = new MassShinglerResult();
            try {
                aggregateShinglerData(context, lists, shingles, ShinglerType.MASS_IN);
            } catch (Exception e) {
                context.session().logger().log(
                    Level.SEVERE,
                    "Failed to aggregate shingler's data for Mass-IN-shingler's request",
                    e);
                callback.completed(null);
                return;
            }
            if (shingles.size() > 0) {
                ImmutableComplaintsConfig complaintsConfig = context.iexProxy().config().complaintsConfig();
                MassShinglerClient client =
                    (MassShinglerClient) complaintsConfig.shinglerClient(ShinglerType.MASS_IN)
                        .adjust(context.session().context());
                if (client == null) {
                    context.session().logger().log(
                        Level.SEVERE,
                        "UpdateDataExecutor.MASS_IN_SHINGLER.updateCounters failed to create client!");
                    callback.completed(null);
                } else {
                    try {
                        final String putQuery = MassShinglerClient.getPutQuery(shingles);
                        context.session().logger().info("Mass-IN-Shingler's putQuery: " + putQuery);
                        if (putQuery == null || putQuery.isEmpty()) {
                            context.session().logger().log(
                                Level.SEVERE,
                                "Failed to generate PUT query from Mass-IN-shingler's data");
                            callback.completed(null);
                        } else if (complaintsConfig.shinglersDryRun()) {
                            context.session().logger().info(
                                "Request to Mass-IN-shingler is omitted due to dry-run mode switched on");
                            callback.completed(null);
                        } else {
                            if (complaintsConfig.shinglersAddStatByActions()
                                && complaintsConfig.shinglersByUserActionStaters().containsKey(ShinglerType.MASS_IN))
                            {
                                client = client.adjustStater(
                                    complaintsConfig.shinglersByUserActionStaters()
                                        .get(ShinglerType.MASS_IN)
                                        .get(context.action()));
                            }
                            client.putShingles(putQuery, callback);
                        }
                    } catch (Exception e) {
                        context.session().logger().log(
                            Level.SEVERE,
                            "Failed to generate PUT query from Mass-IN-shingler's data",
                            e);
                        callback.failed(e);
                    }
                }
            } else {
                callback.completed(null);
            }
        }
    },
    MASS_OUT_SHINGLER {
        @Override
        public void updateCounters(
            final AbstractUserActionContext context,
            final List<UpdateDataHolder> lists,
            final FutureCallback<String> callback,
            final String addonParams)
        {
            final MassShinglerResult shingles = new MassShinglerResult();
            try {
                aggregateShinglerData(context, lists, shingles, ShinglerType.MASS_OUT);
            } catch (Exception e) {
                context.session().logger().log(
                    Level.SEVERE,
                    "Failed to aggregate shingler's data for Mass-OUT-shingler's request",
                    e);
                callback.completed(null);
                return;
            }
            if (shingles.size() > 0) {
                ImmutableComplaintsConfig complaintsConfig = context.iexProxy().config().complaintsConfig();
                MassShinglerClient client =
                    (MassShinglerClient) complaintsConfig.shinglerClient(ShinglerType.MASS_OUT)
                        .adjust(context.session().context());
                if (client == null) {
                    context.session().logger().log(
                        Level.SEVERE,
                        "UpdateDataExecutor.MASS_OUT_SHINGLER.updateCounters failed to create client!");
                    callback.completed(null);
                } else {
                    try {
                        final String putQuery = MassShinglerClient.getPutQuery(shingles);
                        context.session().logger().info("Mass-OUT-shingler's putQuery: " + putQuery);
                        if (putQuery == null || putQuery.isEmpty()) {
                            context.session().logger().log(
                                Level.SEVERE,
                                "Failed to generate PUT query from Mass-OUT-shingler's data");
                            callback.completed(null);
                        } else if (complaintsConfig.shinglersDryRun()) {
                            context.session().logger().info(
                                "Request to Mass-OUT-shingler is omitted due to dry-run mode switched on");
                            callback.completed(null);
                        } else {
                            if (complaintsConfig.shinglersAddStatByActions()
                                && complaintsConfig.shinglersByUserActionStaters().containsKey(ShinglerType.MASS_OUT))
                            {
                                client = client.adjustStater(
                                    complaintsConfig.shinglersByUserActionStaters()
                                        .get(ShinglerType.MASS_OUT)
                                        .get(context.action()));
                            }
                            client.putShingles(putQuery, callback);
                        }
                    } catch (Exception e) {
                        context.session().logger().log(
                            Level.SEVERE,
                            "Failed to generate PUT query from Mass-OUT-shingler's data",
                            e);
                        callback.failed(e);
                    }
                }
            } else {
                callback.completed(null);
            }
        }
    },
    SENDER_SHINGLER {
        @Override
        public void updateCounters(
            final AbstractUserActionContext context,
            final List<UpdateDataHolder> lists,
            final FutureCallback<String> callback,
            final String addonParams)
        {
            final SenderShingles shingles = new SenderShingles();
            try {
                aggregateShinglerData(context, lists, shingles, ShinglerType.SENDER);
            } catch (Exception e) {
                context.session().logger().log(
                    Level.SEVERE,
                    "Failed to aggregate shingler's data for Sender-shingler's request",
                    e);
                callback.completed(null);
                return;
            }
            if (shingles.size() > 0) {
                ImmutableComplaintsConfig complaintsConfig = context.iexProxy().config().complaintsConfig();
                SenderShinglerClient client =
                    (SenderShinglerClient) complaintsConfig.shinglerClient(ShinglerType.SENDER)
                        .adjust(context.session().context());
                if (client == null) {
                    context.session().logger().log(
                        Level.SEVERE,
                        "UpdateDataExecutor.SENDER_SHINGLER.updateCounters failed to create client!");
                    callback.completed(null);
                } else {
                    try {
                        final String putBody = client.getPutQuery(shingles);
                        context.session().logger().info("Sender-shingler's putBody: " + putBody);
                        if (putBody == null || putBody.isEmpty()) {
                            context.session().logger().log(
                                Level.SEVERE,
                                "Failed to generate PUT query from Sender-shingler's data");
                            callback.completed(null);
                        } else if (complaintsConfig.shinglersDryRun()) {
                            context.session().logger().info(
                                "Request to Sender-shingler is omitted due to dry-run mode switched on");
                            callback.completed(null);
                        } else {
                            if (complaintsConfig.shinglersAddStatByActions()
                                && complaintsConfig.shinglersByUserActionStaters().containsKey(ShinglerType.SENDER))
                            {
                                client = (SenderShinglerClient) client
                                    .adjustStater(
                                        complaintsConfig.shinglersByUserActionStaters()
                                            .get(ShinglerType.SENDER)
                                            .get(context.action()));
                            }
                            client.putShingles(putBody, callback);
                            context.session().logger().info("Sender-shingler's PUT done");
                        }
                    } catch (ShingleException e) {
                        context.session().logger().log(
                            Level.SEVERE,
                            "Failed to generate PUT query from Sender-shingler's data",
                            e);
                        callback.failed(e);
                    }
                }
            } else {
                context.session().logger().info("UpdateDataExecutor.SENDER_SHINGLER's request omitted because of "
                    + "empty shingles data");
                callback.completed(null);
            }
        }
    },
    URL_SHINGLER {
        @Override
        public void updateCounters(
            final AbstractUserActionContext context,
            final List<UpdateDataHolder> lists,
            final FutureCallback<String> callback,
            final String addonParams)
        {
            final UrlShingles shingles = new UrlShingles();
            try {
                aggregateShinglerData(context, lists, shingles, ShinglerType.URL);
            } catch (Exception e) {
                context.session().logger().log(
                    Level.SEVERE,
                    "Failed to aggregate shingler's data for Url-shingler's request",
                    e);
                callback.completed(null);
                return;
            }
            if (shingles.size() > 0) {
                ImmutableComplaintsConfig complaintsConfig = context.iexProxy().config().complaintsConfig();
                UrlShinglerClient client =
                    (UrlShinglerClient) complaintsConfig.shinglerClient(ShinglerType.URL)
                        .adjust(context.session().context());
                if (client == null) {
                    context.session().logger().log(
                        Level.SEVERE,
                        "UpdateDataExecutor.URL_SHINGLER.updateCounters failed to create client!");
                    callback.completed(null);
                } else {
                    try {
                        final String putBody = client.getPutQuery(shingles);
                        context.session().logger().info("Url-shingler's putBody: " + putBody);
                        if (putBody == null || putBody.isEmpty()) {
                            context.session().logger().log(
                                Level.SEVERE,
                                "Failed to generate PUT query from Url-shingler's data");
                            callback.completed(null);
                        } else if (complaintsConfig.shinglersDryRun()) {
                            context.session().logger().info(
                                "Request to Url-shingler is omitted due to dry-run mode switched on");
                            callback.completed(null);
                        } else {
                            if (complaintsConfig.shinglersAddStatByActions()
                                && complaintsConfig.shinglersByUserActionStaters().containsKey(ShinglerType.URL))
                            {
                                client = (UrlShinglerClient) client
                                    .adjustStater(
                                        complaintsConfig.shinglersByUserActionStaters()
                                            .get(ShinglerType.URL)
                                            .get(context.action()));
                            }
                            client.putShingles(putBody, callback);
                        }
                    } catch (ShingleException e) {
                        context.session().logger().log(
                            Level.SEVERE,
                            "Failed to generate PUT query from Url-shingler's data",
                            e);
                        callback.failed(e);
                    }
                }
            } else {
                callback.completed(null);
            }
        }
    };

    private static Map<String, Object> mkFunc(final String funcName, final Object... args) {
        return Map.of(
            "function", funcName,
            "args", Arrays.asList(args));
    }

    public static String abusesUrl(final String uid, final String stid, final String abuseType) {
        return "abuses_" + uid + '_' + stid + '_' + abuseType;
    }

    public abstract void updateCounters(
        final AbstractUserActionContext context,
        final List<UpdateDataHolder> lists,
        final FutureCallback<String> callback,
        final String addonParams);

    public void updateDocs(
        final AbstractUserActionContext context,
        final List<Map<String, Object>> docs,
        final FutureCallback<String> callback,
        final String addonParams)
    {
        if (docs == null || docs.isEmpty() || context.uid().isEmpty()) {
            context.session().logger().info("UpdateDataExecutor.updateDocs skipped for " + name());
            callback.completed(null);
            return;
        }
        final String countersType = name().toLowerCase(Locale.ROOT);
        final String body = JsonType.DOLLAR.toString(
            Map.of(
                "AddIfNotExists", true,
                "prefix", context.uid(),
                "docs", docs));
        HttpHost host = context.iexProxy().config().producerAsyncClientConfig().host();

        BasicAsyncRequestProducerGenerator generator =
            new BasicAsyncRequestProducerGenerator(
                "/update?" + countersType + "&prefix=" + context.uid() + addonParams,
                body);
        generator.addHeader(YandexHeaders.SERVICE, context.mailSearchQueueName());

        AsyncClient producerAsyncClient =
            context.iexProxy().producerAsyncClient().adjust(context.session().context());

        producerAsyncClient.execute(
            host,
            generator,
            AsyncStringConsumerFactory.OK,
            context.session().listener().createContextGeneratorFor(producerAsyncClient),
            new AbstractFilterFutureCallback<>(callback) {
                @Override
                public void completed(String s) {
                    context.session().logger().info("UpdateDataExecutor." + name() + ": counters updated successfully");
                    callback.completed(s);
                }
            });
    }

    private static void aggregateShinglerData(
        final AbstractUserActionContext context,
        final List<UpdateDataHolder> lists,
        final AbstractShinglesMap<?> shingles,
        final ShinglerType shinglerType)
    {
        try {
            for (final UpdateDataHolder updateDataHolder : lists) {
                if (updateDataHolder.containsShinglerData(shinglerType)) {
                    try {
                        shingles.addAll(updateDataHolder.shinglerData(shinglerType));
                        if (shinglerType == ShinglerType.ACTIVITY || shinglerType == ShinglerType.COMPL
                                || shinglerType == ShinglerType.SENDER || shinglerType == ShinglerType.URL)
                        {
                            context.session().logger().info("UpdateDataExecutor.aggregateShinglerData: shingles="
                                + shingles);
                        }
                    } catch (ShingleException e) {
                        context.session().logger().log(
                            Level.SEVERE,
                            "Failed to gather " + shinglerType.name().toLowerCase(Locale.ROOT) + "-shingler's data",
                            e);
                    }
                } else {
                    context.session().logger().info(shinglerType.name() + " shingler has no data!");
                }
            }
        } catch (Exception e) {
            context.session().logger().log(Level.SEVERE, "agregateShinglerData exception: " + e, e);
        }
    }
}
