package ru.yandex.iex.proxy.refundhandler;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Map;
import java.util.Map.Entry;
import java.util.TimeZone;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.http.HttpHeaders;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.ContentType;

import ru.yandex.base64.Base64Encoder;
import ru.yandex.function.ByteArrayProcessable;
import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.ServerException;
import ru.yandex.http.util.YandexHttpStatus;
import ru.yandex.http.util.nio.AsyncStringConsumerFactory;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.iex.proxy.IexProxy;
import ru.yandex.iex.proxy.xutils.spamreport.SpamReportSender;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.dom.JsonString;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonWriter;

public class EmlTransactionsCallback
    extends AbstractProxySessionCallback<ByteArrayProcessable>
{
    public static final TimeZone TIMEZONE = TimeZone.getTimeZone("Europe/Moscow");
    private final ProxySession session;
    private ArrayList<Map<Object,Object>> transactionsToTheNextHop;
    private RefundContext initialContext;
    private boolean isFraud;
    private boolean known;
    private String parseAsHtml;
    private String bank;
    private IexProxy iexProxy;

    EmlTransactionsCallback(
            final ProxySession session,
            final ArrayList<Map<Object, Object>> transactionsToTheNextHop,
            final RefundContext initialContext,
            final boolean known,
            final String parseAsHtml,
            final boolean isFraud,
            final String bank,
            final IexProxy iexProxy)
    {
        super(session);
        this.session = session;
        this.transactionsToTheNextHop = transactionsToTheNextHop;
        this.initialContext = initialContext;
        this.known = known;
        this.parseAsHtml = parseAsHtml;
        this.bank = bank;
        this.isFraud = isFraud;
        this.iexProxy = iexProxy;
    }

    @Override
    public void completed(final ByteArrayProcessable result) {
        StringBuilderWriter sbw = new StringBuilderWriter(new StringBuilder(""));
        try (JsonWriter writer = new JsonWriter(sbw)) {
            writer.startObject();
            writer.key("mid");
            writer.value("" + initialContext.getMid());
            writer.key("known");
            writer.value(known);
            writer.key("parse_as_html");
            writer.value(parseAsHtml);
            writer.key("mds_link");
            writer.value(initialContext.getStid());
            writer.key("is_fraud");
            writer.value(isFraud);
            String subject = session.params().getOrNull("subject");
            if (subject.contains("ANTIFRAUD")) {
                Pattern ticketIdPattern = Pattern.compile(".*(ANTIFRAUD-\\d+).*");
                Matcher ticketIdMatcher = ticketIdPattern.matcher(subject);
                if (ticketIdMatcher.find()) {
                    writer.key("ticket_id");
                    writer.value(ticketIdMatcher.group(1));
                }
            }
            writer.key("transactions");
            writer.startArray();
            for (Map<Object,Object> transactionInfo: transactionsToTheNextHop) {
                writer.startObject();
                for (Map.Entry<Object,Object> transactionEntry : transactionInfo.entrySet()) {
                    writer.key(transactionEntry.getKey().toString());
                    writer.value(transactionEntry.getValue());
                }
                writer.key("dt_mail_ts");
                writer.value("" + initialContext.getReceivedDate());
                writer.key("dt_mail");
                writer.value(
                        DateTimeFormatter.ISO_LOCAL_DATE.format(
                            LocalDateTime.ofInstant(
                                Instant.ofEpochSecond(initialContext.getReceivedDate()),
                                TIMEZONE.toZoneId())));
                writer.key("bank");
                writer.value(bank);
                writer.endObject();
            }
            writer.endArray();
            ByteArrayOutputStream baos = new ByteArrayOutputStream();
            result.processWith(SpamReportSender.ByteArrayBodyFactory.INSTANCE).writeTo(baos);
            Base64Encoder encoder = new Base64Encoder();
            encoder.process(baos.toByteArray());
            writer.key("eml");
            writer.value(new JsonString(encoder.toString()));
            writer.endObject();

            BasicAsyncRequestProducerGenerator generator =
                new BasicAsyncRequestProducerGenerator("/request", sbw.toString());
            generator.addHeader(
                HttpHeaders.CONTENT_TYPE,
                ContentType.APPLICATION_JSON.toString());
            generator.addHeader(
                HttpHeaders.ACCEPT,
                ContentType.APPLICATION_JSON.toString());

            session.logger().info("EmlTransactionsCallback: sending data: "
                    + JsonType.NORMAL.toString(sbw.toString()));
            AsyncClient refundClient =
                iexProxy.refundClient().adjust(
                session.context());
            refundClient.execute(
                iexProxy.config().refundConfig().host(),
                generator,
                AsyncStringConsumerFactory.OK,
                session.listener().
                    createContextGeneratorFor(refundClient),
               new RefundCallback());
        } catch (IOException e) {
            session.logger().info("EmlTransactionsCallback: exception while constructing transactions data");
            this.failed(e);
        }
    }

    private class RefundCallback implements FutureCallback<String> {
        @Override
        public void cancelled() {
            session.logger().info("RefundCallback: cancelled");
            session.logger().info("RefundCallback:"
                    + " uid = " + initialContext.uid()
                    + " mid = " + initialContext.getMid() + " cancelled");
            this.cancelled();
        }

        @Override
        public void failed(final Exception e) {
            StringWriter sw = new StringWriter();
            e.printStackTrace(new PrintWriter(sw));
            if (e instanceof ServerException) {
                ServerException se = (ServerException) e;
                session.logger().info("RefundCallback:"
                    + " uid = " + initialContext.uid()
                    + " mid = " + initialContext.getMid()
                    + " ServerException " + se.statusCode());
                if (se.statusCode() == YandexHttpStatus.SC_ACCEPTED) {
                    session.logger().info("RefundCallback: mid is procesing in another req, skipping");
                    initialContext.response(se.statusCode(), "OK, is processing");
                } else if (se.statusCode() == 208) {
                    session.logger().info("RefundCallback: mid was processed in another req, skipping");
                    initialContext.response(se.statusCode(), "OK, already processed");
                } else {
                    session.logger().info("RefundCallback: mid failed with " + se.getMessage());
                    initialContext.response(se.statusCode(), se.getMessage());
                }
            } else {
                session.logger().info("RefundCallback:"
                    + " uid = " + initialContext.uid()
                    + " mid = " + initialContext.getMid()
                    + " failed with " + sw.toString());
                initialContext.response(YandexHttpStatus.SC_GATEWAY_TIMEOUT, "default, gateway timeout");
            }
        }

        @Override
        public void completed(final String result) {
            session.logger().info("RefundCallback:"
                + " uid = " + initialContext.uid()
                + " mid = " + initialContext.getMid()
                + " completed with: <" + result + '>');
            initialContext.response();
        }
    }
}
