package ru.yandex.iex.proxy.taxihandler;

import java.util.AbstractMap;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.logging.Level;

import org.apache.http.HttpException;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.geocoder.GeocoderClient;
import ru.yandex.geocoder.GeocoderRequest;
import ru.yandex.geocoder.GeocoderResult;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.HttpExceptionConverter;
import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.http.util.ServerException;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.YandexHttpStatus;
import ru.yandex.iex.proxy.AbstractEntityHandler;
import ru.yandex.iex.proxy.IexProxy;
import ru.yandex.iex.proxy.XJsonUtils;
import ru.yandex.iex.proxy.XMessageToLog;
import ru.yandex.iex.proxy.XRegexpUtils;
import ru.yandex.json.xpath.JsonUnexpectedTokenException;
import ru.yandex.stater.RequestInfo;
import ru.yandex.util.timesource.TimeSource;

public class TaxiHandler extends AbstractEntityHandler<TaxiContext> {
    public TaxiHandler(final IexProxy iexProxy) {
        super(iexProxy, "taxi");
    }

    @Override
    protected TaxiContext createContext(
            final IexProxy iexProxy,
            final ProxySession session,
            final Map<?, ?> json)
        throws HttpException, JsonUnexpectedTokenException
    {
        return new TaxiContext(iexProxy, session, json);
    }

    @Override
    protected void handle(final TaxiContext context)
        throws JsonUnexpectedTokenException, BadRequestException
    {
        Map<?, ?> firstTaxiNode = context.firstNodeOftaxiJson();
        Object arrCity = null;
        Object depCity = null;
        if (firstTaxiNode != null) {
            arrCity = firstTaxiNode.get("arr");
            depCity = firstTaxiNode.get("dep");
        }
        if (arrCity instanceof String && depCity instanceof String) {
            GeocoderClient client =
                context.iexProxy().geoSearchClient().adjust(
                    context.session().context());
            final MultiFutureCallback<ProxyResult> multiCallback =
                new MultiFutureCallback<>(
                    new FinishMultiCallback(context));
            final List<ResultCallback> subCallbacks =
                new ArrayList<>(2);
            subCallbacks.add(
                   new ResultCallback(
                       context,
                       multiCallback.newCallback(),
                       (String) arrCity).setFutureKey("geocoder_arr_city"));
            subCallbacks.add(
                new ResultCallback(
                    context,
                    multiCallback.newCallback(),
                    (String) depCity).setFutureKey("geocoder_dep_city"));

            multiCallback.done();
            try {
                for (ResultCallback x : subCallbacks) {
                    GeocoderRequest request = new GeocoderRequest(
                        context.iexProxy().config().geoSearchConfig());
                    request.addHeader(
                        YandexHeaders.X_YA_SERVICE_TICKET,
                        context.iexProxy().geoTvm2Ticket());
                    request.setText(x.getQuery());
                    context.session().logger().info(
                        "Send geocoder request: " + request.getQuery());
                    client.execute(
                        request,
                        context.session().listener()
                            .createContextGeneratorFor(client),
                        x);
                }
            } catch (BadRequestException e) {
                oldHandleProcess(context);
            }
        } else {
            oldHandleProcess(context);
        }
    }

    class FinishMultiCallback
        implements FutureCallback<List<ProxyResult>>
    {
        private final TaxiContext context;

        FinishMultiCallback(final TaxiContext context) {
            this.context = context;
        }

        @Override
        public void completed(final List<ProxyResult> results) {
            for (ProxyResult x : results) {
                if (x != null) {
                    try {
                        XJsonUtils.pushToMap(
                            context.firstNodeOftaxiJson(),
                            x.getFutureKeyName(),
                            x.getCity());
                        Iterator<Map.Entry<String, Object>> it = x.iterator();
                        while (it.hasNext()) {
                            Map.Entry<String, Object> y = it.next();
                            if (y != null) {
                                XJsonUtils.pushToMap(
                                    context.firstNodeOftaxiJson(),
                                    y.getKey(),
                                    y.getValue());
                            }
                        }
                    } catch (
                        JsonUnexpectedTokenException
                            | NoSuchElementException e)
                    {
                    }
                }
            }
            oldHandleProcess(context);
        }

        @Override
        public void cancelled() {
            oldHandleProcess(context);
        }

        @Override
        public void failed(final Exception e) {
            oldHandleProcess(context);
        }
    }

    private static class ProxyResult {
        private final String city;
        private final String futureKeyName;
        private final ArrayList<Map.Entry<String, Object>> fields
            = new ArrayList<>();

        ProxyResult(final String city, final String key) {
            this.city = city;
            futureKeyName = key;
        }

        public String getCity() {
            return city;
        }

        public String getFutureKeyName() {
            return futureKeyName;
        }

        public void put(final String key, final Object value) {
            fields.add(new AbstractMap.SimpleEntry<>(key, value));
        }

        public Iterator<Map.Entry<String, Object>> iterator() {
            return new Iterator<Map.Entry<String, Object>>() {
                private int curId = 0;
                @Override
                public boolean hasNext() {
                    return curId < fields.size();
                }

                @Override
                public Map.Entry<String, Object> next()
                    throws NoSuchElementException
                {
                    if (curId >= fields.size()) {
                        throw new NoSuchElementException();
                    }
                    return fields.get(curId++);
                }
            };
        }
    }

    private static class ResultCallback
        implements FutureCallback<GeocoderResult>
    {
        private final long start = TimeSource.INSTANCE.currentTimeMillis();
        private final FutureCallback<ProxyResult> callback;
        private final TaxiContext context;
        private final String query;
        private String futureKeyName = "geocoder_city";

        ResultCallback(
            final TaxiContext context,
            final FutureCallback<ProxyResult> callback,
            final String query)
        {
            this.callback = callback;
            this.context = context;
            this.query = query;
        }

        ResultCallback setFutureKey(final String key) {
            futureKeyName = key;
            return this;
        }

        public String getQuery() {
            return query;
        }

        @Override
        public void completed(final GeocoderResult result) {
            stat(YandexHttpStatus.SC_OK);
            if (!result.getCities().isEmpty()) {
                int precision = 0;
                double lat = (result.lowerLatitude(precision)
                    + result.upperLatitude(precision)) / 2.0;
                double lon = (result.lowerLongitude(precision)
                    + result.upperLongitude(precision)) / 2.0;
                ProxyResult res = new ProxyResult(
                    result.getCities().get(0), futureKeyName);
                res.put(futureKeyName + "_lat", lat);
                res.put(futureKeyName + "_lon", lon);
                callback.completed(res);
            }
            callback.completed(null);
        }

        @Override
        public void cancelled() {
            stat(YandexHttpStatus.SC_CLIENT_CLOSED_REQUEST);
            context.session()
                .logger()
                .warning("Request cancelled: "
                    + this.context.session().listener().details());
            context.session().response(
                YandexHttpStatus.SC_CLIENT_CLOSED_REQUEST);
        }

        @Override
        public void failed(final Exception e) {
            if (e instanceof ServerException) {
                stat(((ServerException) e).statusCode());
            } else {
                stat(YandexHttpStatus.SC_REMOTE_CLOSED_REQUEST);
            }
            context.session()
                .logger()
                .log(Level.WARNING, "Failed to process: "
                    + context.humanReadableJson()
                    + '\n' + context.session().listener().details()
                    + " because of exception", e);
            context.session().handleException(
                HttpExceptionConverter.toHttpException(e));
        }

        private void stat(final int status) {
            context.iexProxy().geoRequestsStater().accept(
                new RequestInfo(
                    TimeSource.INSTANCE.currentTimeMillis(),
                    status,
                    start,
                    start,
                    0L,
                    0L));
        }
    }

    private void oldHandleProcess(final TaxiContext context) {
        Map<?, ?> firstTaxiNode = context.firstNodeOftaxiJson();
        Object protoCoords = null;
        if (firstTaxiNode != null) {
            protoCoords = firstTaxiNode.get("crfrom");
            if (protoCoords == null) {
                protoCoords = firstTaxiNode.get("crto");
            }
        }
        String coords = "";
        if (protoCoords instanceof String) {
            coords = (String) protoCoords;
        }
        if (!coords.isEmpty()) { // only gett.com can include crfrom and crto
            ArrayList<String> l = XRegexpUtils.getDoubles(coords);
            if (l.size() >= 2) {
                String ll = l.get(0) + ',' + l.get(1);
                GeocoderClient client =
                    context.iexProxy().geoSearchClient().adjust(
                        context.session().context());
                try {
                    GeocoderRequest request = new GeocoderRequest(
                        context.iexProxy().config().geoSearchConfig());
                    request.addHeader(
                        YandexHeaders.X_YA_SERVICE_TICKET,
                        context.iexProxy().geoTvm2Ticket());
                    request.setLl(ll);
                    context.session().logger().info(
                        "Send ll geocoder request: " + request.getQuery());
                    client.execute(
                        request,
                        context.session().listener()
                            .createContextGeneratorFor(client),
                        new TaxiCallback(context)
                    );
                } catch (BadRequestException e) {
                    context.session().logger().log(
                        Level.SEVERE,
                        "GeoCoder request encoding failed",
                        e);
                    context.response();
                }
            } else {
                XMessageToLog.warning(
                    context,
                    "Coords lon and lat are not found in the crfrom.");
                context.response();
            }
        } else {
            context.response();
        }
    }
}
