package ru.yandex.iex.proxy.tickethandlerlegacy;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.logging.Level;
import java.util.regex.Pattern;

import org.apache.http.HttpException;

import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.HttpExceptionConverter;
import ru.yandex.http.util.YandexHttpStatus;
import ru.yandex.iex.proxy.AbstractEntityContext;
import ru.yandex.iex.proxy.IexProxy;
import ru.yandex.iex.proxy.IndexationContext;
import ru.yandex.iex.proxy.XJsonUtils;
import ru.yandex.iex.proxy.XMessageToLog;
import ru.yandex.iex.proxy.XRegexpUtils;
import ru.yandex.iex.proxy.XStrAlgo;
import ru.yandex.iex.proxy.XTimeUtils;
import ru.yandex.iex.proxy.flightextractor.FlightExtractor;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.xpath.JsonUnexpectedTokenException;
import ru.yandex.json.xpath.ValueUtils;
import ru.yandex.parser.email.types.MessageType;
import ru.yandex.stater.RequestsStater;

public class TicketContext extends AbstractEntityContext {
    public static final String BOOKING_TODAY = "booking-today";
    public static final String CHECKIN_URL = "checkin_url";
    public static final String TICKET_NUMBER = "ticketNumber";
    public static final String TICKET_UNDERLINE_NUMBER = "ticket_number";
    public static final String IEX_PATTERNS = "iex-patterns";
    public static final String MICRODATA = "microdata";
    public static final String RASP = "rasp";
    public static final String COMMA = ",";
    public static final String EMAIL = "email";
    public static final String USER_EMAIL = "user_email";
    public static final String PHONE = "phone";
    public static final String DOMAIN = "domain";
    public static final String FLIGHT_TYPE = "@type";
    public static final String CHANGE_STATUS = "change_status";
    public static final String PAYMENT_STATUS = "payment_status";
    public static final String PDF = "pdf";
    public static final String HID = "hid_";
    public static final String BACK = "_back";
    public static final String TICKET = "ticket";
    public static final String MICRO = "micro";
    public static final String UNKNOWN = "unknown";
    public static final String ORIGIN = "origin";
    public static final String DATE = "date_";
    public static final String NUMBER = "number_";
    public static final String TITLE_PARTS = "title_parts";
    public static final String SCHEDULED_ARRIVAL = "scheduled_arrival";
    public static final String SCHEDULED_DEPARTURE = "scheduled_departure";
    public static final String REGEXP_P = "regexp_";
    public static final String HTML = "html";
    public static final String IEX_MSG = "iex_msg";
    public static final String REGEXP_EXTRACTION = "regexp_extraction";
    public static final String SPACE_AS_STR = " ";
    public static final String PDF_ORIGIN_ID = REGEXP_P + PDF;
    public static final String RASP_PDF_ORIGIN_ID = RASP + '_' + PDF_ORIGIN_ID;
    public static final String HTML_ORIGIN_ID = REGEXP_P + HTML;
    public static final String RASP_HTML_ORIGIN_ID =
        RASP + '_' + HTML_ORIGIN_ID;
    public static final String IATA = "_iata";
    public static final int MAX_POSSIBLE_CFLIGHT = 32;
    public static final String LAST = "_last";
    public static final String XPATH_XPATH = "xpath_xpath";
    public static final String UNDEFINED = "undefined";
    public static final String URL = "url";
    public static final String REMINDER_TYPE = "63";
    public static final String MICROHTML = "microhtml";
    public static final String REMINDER = "reminder";
    public static final String JETSTAR_COM = "jetstar.com";
    private static final String TRANSFER = "transfer";
    private static final String TYPE_FROM_EXTRACTOR = "type_from_extractor";
    private static final Pattern TIME_DATE_RASP_PATTERN =
        Pattern.compile("20[0-9]{2}-[0-9]{1,2}-[0-9]{1,2}"
            + "T[0-9]{1,2}:[0-9]{1,2}:[0-9]{1,2}");
    private static final Pattern PAYMENT_STATUS_UNPAID_PATTERN =
        Pattern.compile("[0-9]{1,2} (января|февраля|марта|апреля|мая|июня|июля|"
            + "августа|сентября|октября|ноября|декабря) [0-9]{1,2}:[0-9]{2} "
            + "\\(\\+[0-9]{2}:[0-9]{2}\\)");
    private static final ArrayList<RaspEntry> RASP2IEX;
    private static String[] times = {
        TicketIEX.DATE_DEP,
        TicketIEX.DATE_ARR,
        TicketIEX.DATE_DEP_BACK,
        TicketIEX.DATE_DEP_LAST,
        TicketIEX.TIME_DEP,
        TicketIEX.TIME_ARR,
        TicketIEX.TIME_DEP_BACK,
        TicketIEX.TIME_DEP_LAST,
        TicketIEX.TIME_ARR_BACK,
        TicketIEX.DATE_ARR_BACK
    };

    static {
        RASP2IEX = new ArrayList<>();
        RASP2IEX.add(new RaspEntry(TicketIEX.FLIGHT_NUMBER, "number"));
        RASP2IEX.add(new RaspEntry(TicketIEX.AIRLINE, "company"));
        RASP2IEX.add(new RaspEntry(TicketIEX.AIRLINE_IATA, "company_iata"));
        RASP2IEX.add(new RaspEntry(TicketIEX.CITY_DEP, "title_parts_0"));
        RASP2IEX.add(
            new RaspEntry(TicketIEX.CITY_DEP_GEOID, "from_settlement_geoid"));
        RASP2IEX.add(new RaspEntry(TicketIEX.AIRPORT_DEP, "from_name"));
        RASP2IEX.add(new RaspEntry(TicketIEX.AIRPORT_DEP_IATA, "from_iata"));
        RASP2IEX.add(new RaspEntry(TicketIEX.CITY_ARR, "title_parts_1"));
        RASP2IEX.add(
            new RaspEntry(TicketIEX.CITY_ARR_GEOID, "to_settlement_geoid"));
        RASP2IEX.add(new RaspEntry(TicketIEX.AIRPORT_ARR, "to_name"));
        RASP2IEX.add(new RaspEntry(TicketIEX.AIRPORT_ARR_IATA, "to_iata"));
        RASP2IEX.add(new RaspEntry(TicketIEX.DATE_DEP, SCHEDULED_DEPARTURE));
        //RASP2IEX.add(new RaspEntry(TicketIEX.TIME_DEP, SCHEDULED_DEPARTURE));
        RASP2IEX.add(new RaspEntry(TicketIEX.DATE_ARR, SCHEDULED_ARRIVAL));
        //RASP2IEX.add(new RaspEntry(TicketIEX.TIME_ARR, SCHEDULED_ARRIVAL));
    }

    //aeroflot's methods and data
    private static final Map<String, Object> MICRO2IEX;

    static {
        MICRO2IEX = new HashMap<String, Object>();
        MICRO2IEX.put(
            TicketAeroflot.FLIGHTNUMBER,
            TicketIEX.FLIGHT_NUMBER);
        MICRO2IEX.put(
            ru.yandex.iex.proxy.tickethandlerlegacy
                .TicketAeroflot.DEPARTURETIME,
            new ArrayList<String>(
                Arrays.asList(
                    TicketIEX.TIME_DEP,
                    TicketIEX.DATE_DEP,
                    TicketIEX.DATE_DEP_TZ)));
        MICRO2IEX.put(
            TicketAeroflot.ARRIVALTIME,
            new ArrayList<String>(
                Arrays.asList(
                    TicketIEX.TIME_ARR,
                    TicketIEX.DATE_ARR,
                    TicketIEX.DATE_ARR_TZ)));

        MICRO2IEX.put(
            ru.yandex.iex.proxy.tickethandlerlegacy
                .TicketAeroflot.ARRIVALAIRPORT
            + TicketAeroflot.IATACODE,
            TicketIEX.AIRPORT_ARR_IATA);
        MICRO2IEX.put(
            ru.yandex.iex.proxy.tickethandlerlegacy
                .TicketAeroflot.ARRIVALAIRPORT
            + ru.yandex.iex.proxy.tickethandlerlegacy
                .TicketAeroflot.ADDRESSLOCALITY,
            TicketIEX.CITY_ARR);

        MICRO2IEX.put(
            ru.yandex.iex.proxy.tickethandlerlegacy
                .TicketAeroflot.DEPARTUREAIRPORT
            + TicketAeroflot.IATACODE,
            TicketIEX.AIRPORT_DEP_IATA);
        MICRO2IEX.put(
            TicketAeroflot.DEPARTUREAIRPORT
            + TicketAeroflot.ADDRESSLOCALITY,
            TicketIEX.CITY_DEP);

        MICRO2IEX.put(
            TicketAeroflot.AIRLINE
            + TicketAeroflot.IATACODE,
            TicketIEX.AIRLINE_IATA);

        MICRO2IEX.put(
            TicketAeroflot.ARRIVALAIRPORT
                + TicketAeroflot.NAME,
            TicketIEX.AIRPORT_ARR);

        MICRO2IEX.put(
            TicketAeroflot.DEPARTUREAIRPORT
                + TicketAeroflot.NAME,
            TicketIEX.AIRPORT_DEP);

        MICRO2IEX.put(
            TicketAeroflot.ARRIVALAIRPORT
                + TicketAeroflot.ADDRESS
                + TicketAeroflot.ADDRESSLOCALITY,
            TicketIEX.CITY_ARR);

        MICRO2IEX.put(
            TicketAeroflot.DEPARTUREAIRPORT
                + TicketAeroflot.ADDRESS
                + TicketAeroflot.ADDRESSLOCALITY,
            TicketIEX.CITY_DEP);
        MICRO2IEX.put("webCheckinUrl", CHECKIN_URL);
        MICRO2IEX.put("checkinUrl", CHECKIN_URL);
    }

    private ArrayList<Map<String, Object>> ticketJson;
    private Map<String, Object> ptrToMainTicketJson;
    private List<String> pdfParts;
    private TicketFlightSet uniqFlights;
    private ArrayList<String> raspData
        = new ArrayList<String>();
    private RequestsStater raspStater;
    private String mid;
    private String uid;
    private String email = "";
    private String from = "";
    private String domain = "";
    private String subject = "";
    private long receivedDate = 0L;
    private boolean isCalled = false;
    private boolean bookingStatus = false;
    private Map<?, ?> inputJson = null;
    private Set<Object> ticketNumbersFromMicro = new HashSet<>();
    //private XStrAlgo.HtmlSanitizer
    // htmlSanitizer = new XStrAlgo.HtmlSanitizer();

    private abstract class TicketUnit {
        protected boolean status = false;

        abstract void parse(final Map<?, ?> json, final boolean prevUnitsStatus)
            throws JsonUnexpectedTokenException;

        abstract String getOrigin();

        public boolean getStatus() {
            return status;
        }

        public void useExternalFlightExtractor(final Map<?, ?> fm) {
            if (fm != null) {
                FlightExtractor f = new FlightExtractor(
                    domain,
                    receivedDate,
                    fm);
                if (f.hasFlight()) {
                    ticketJson.addAll(f.getArrayOfFlights());
                }
            }
        }

        protected void completeDateAndTime(final Map<String, Object> ticket) {
            Object dateDep = ticket.get(TicketIEX.DATE_DEP);
            Object timeDep = ticket.get(TicketIEX.TIME_DEP);
            if (dateDep != null && timeDep == null) {
                ticket.put(TicketIEX.TIME_DEP, dateDep);
            }
            if (dateDep == null && timeDep != null) {
                ticket.put(TicketIEX.DATE_DEP, timeDep);
            }
        }
    }

    private class MicroUnit extends TicketUnit {
        @Override
        void parse(final Map<?, ?> json, final boolean microStatus)
            throws JsonUnexpectedTokenException
        {
            if (domain.equals(JETSTAR_COM)) { // turn micro for specific url
                return;
            }
            status = parseMicroData(
                 asMapOrNullWithCheck(json, MICRO),
                 getOrigin());
        }

        @Override
        String getOrigin() {
            return MICRO;
        }
    }

    private class MicroHtmlUnit extends TicketUnit {
        @Override
        void parse(final Map<?, ?> json, final boolean microStatus)
            throws JsonUnexpectedTokenException
        {
            status = parseMicroData(
                asMapOrNullWithCheck(json, MICROHTML),
                getOrigin());
        }

        @Override
        String getOrigin() {
            return MICRODATA;
        }
    }

    private class IexPatternsUnit extends TicketUnit {
        @Override
        void parse(final Map<?, ?> json, final boolean microStatus)
            throws JsonUnexpectedTokenException
        {
            if (!microStatus) {
                status = parseIexPatternsData(
                    asMapOrNullWithCheck(json, TICKET),
                    getOrigin());
            }
        }

        @Override
        String getOrigin() {
            return IEX_PATTERNS;
        }
    }

    private class PdfTicketUnit extends TicketUnit {
        @Override
        void parse(final Map<?, ?> json, final boolean microStatus)
            throws JsonUnexpectedTokenException
        {
            status = parsePdfData(json, microStatus, getRegexpOrigin());
        }

        @Override
        String getOrigin() {
            return PDF;
        }

        public String getRegexpOrigin() {
            return PDF_ORIGIN_ID;
        }
    }

    private class XpathUnit extends TicketUnit {
        @Override
        void parse(final Map<?, ?> json, final boolean microStatus)
            throws JsonUnexpectedTokenException
        {
            Object xpath =
                XJsonUtils.getNodeByPathOrNull(json, TICKET, XPATH_XPATH);
            if (xpath instanceof Map) {
                HashMap<String, Object> fm = new HashMap<>();
                XJsonUtils.putAll(xpath, fm);
                completeDateAndTime(fm);
                useExternalFlightExtractor(fm);
            }
        }

        @Override
        String getOrigin() {
            return "xpath";
        }
    }

    private class XpathRegexp extends TicketUnit {
        public static final String XPATH_REGEXP = "xpath_regexp";
        public static final String DEP_DATE = "dep_date";
        public static final int MAX_DATES = 4;

        @Override
        void parse(final Map<?, ?> json, final boolean microStatus)
            throws JsonUnexpectedTokenException
        {
            Object ticket = XJsonUtils.getNodeByPathOrNull(
                json,
                TICKET,
                XPATH_REGEXP);
            if (ticket instanceof Map) {
                HashMap<String, Object> subTicketJson = new HashMap<>();
                XJsonUtils.putAll(ticket, subTicketJson);
                parseDepDate(ticket, subTicketJson);
                completeDateAndTime(subTicketJson);
                subTicketJson.put(ORIGIN, getOrigin());
                ticketJson.add(subTicketJson);
                useExternalFlightExtractor((Map<?, ?>) ticket);
            }
        }

        @Override
        String getOrigin() {
            return XPATH_REGEXP;
        }

        private void parseDepDate(
            final Object in,
            final HashMap<String, Object> subTicketJson)
        {
            try {
                Map<?, ?> inJson = ValueUtils.asMap(in);
                if (inJson.containsKey(DEP_DATE)) {
                    StringBuilder sb = new StringBuilder();
                    sb.append((String) inJson.get(DEP_DATE));
                    sb.append(' ');
                    if (inJson.containsKey(TicketIEX.DATE_DEP)) {
                        sb.append((String) inJson.get(TicketIEX.DATE_DEP));
                    }
                    String[] rawDates = sb.toString().split("\\s");
                    if (rawDates.length >= MAX_DATES) {
                        StringBuilder sbfinalDate = new StringBuilder();
                        sbfinalDate.append(rawDates[1]);
                        sbfinalDate.append(' ');
                        sbfinalDate.append(rawDates[2 + 1]);
                        subTicketJson.put(
                            TicketIEX.DATE_DEP,
                            sbfinalDate.toString());
                    }
                }
            } catch (JsonUnexpectedTokenException e) {
            }
        }
    }

    private class RegexpHtmlUnit extends TicketUnit {
        @Override
        void parse(final Map<?, ?> json, final boolean microStatus)
            throws JsonUnexpectedTokenException
        {
            boolean extracted = false;
            for (Map<String, Object> ticket: ticketJson) {
                if (TicketFlightSet.checkMinimumRequiredSet(ticket)) {
                    extracted = true;
                    break;
                }
            }
            if (!microStatus && !extracted) {
                parseIexFlightAndDate(json, HTML_ORIGIN_ID);
            }
        }

        @Override
        String getOrigin() {
            return HTML_ORIGIN_ID;
        }
    }

    private ArrayList<TicketUnit> ticketsUnits =
        new ArrayList<>(Arrays.asList(
            new MicroUnit(),
            new MicroHtmlUnit(),
            new IexPatternsUnit(),
            new PdfTicketUnit(),
            new XpathUnit(),
            new XpathRegexp(),
            new RegexpHtmlUnit()));

    public TicketContext(
        final IexProxy iexProxy,
        final ProxySession session,
        final Map<?, ?> json)
        throws HttpException, JsonUnexpectedTokenException
    {
        super(iexProxy, session, json);
        try {
            if (json == null) {
                throw new BadRequestException("No json supplied");
            } else {
                ticketJson = new ArrayList<>();
                preParseAction(json);
                parseTicketInfo(json);
                inputJson = json;
                raspStater = iexProxy.raspRequestsStater();
            }
        } catch (NullPointerException e) {
            throw new BadRequestException("NPE happened when parsing ticket");
        }
    }

    public ArrayList<Map<String, Object>> getTicketJson() {
        return ticketJson;
    }

    public RequestsStater getRaspStater() {
        return raspStater;
    }

    public long getReceivedDate() {
        return receivedDate;
    }

    public String getDomain() {
        return domain;
    }

    public String getMid() {
        return mid;
    }

    public String uid() {
        return uid;
    }

    public String getEmail() {
        return email;
    }

    public String getFrom() {
        return from;
    }

    private static class RaspEntry {
        private String a;
        private String b;

        RaspEntry(final String a, final String b) {
            this.a = a;
            this.b = b;
        }

        public String first() {
            return a;
        }

        public String second() {
            return b;
        }
    }

    public void renameRaspFields(final Map<String, Object> m) {
        if (m.containsKey(TITLE_PARTS)) {
            Object lr = m.get(TITLE_PARTS);
            if (lr instanceof List) {
                List<?> li = (List<?>) lr;
                if (li.size() >= 2) {
                    m.put(TicketIEX.CITY_DEP, li.get(0));
                    m.put(TicketIEX.CITY_ARR, li.get(1));
                }
            }
            m.remove(TITLE_PARTS);
        }
        for (final RaspEntry x : RASP2IEX) {
            String raspKey = x.second();
            if (m.containsKey(raspKey)) {
                Object value = m.get(raspKey);
                m.remove(raspKey);
                m.put(x.first(), value);
                if (x.first().equals(TicketIEX.DATE_DEP)) {
                    m.put(TicketIEX.TIME_DEP, value);
                    m.put(TicketIEX.DATE_DEP_TZ, value);
                }
                if (x.first().equals(TicketIEX.DATE_ARR)) {
                    m.put(TicketIEX.TIME_ARR, value);
                    m.put(TicketIEX.DATE_ARR_TZ, value);
                }
            }
        }
        if (m.containsKey(TicketIEX.CITY_DEP)
            && m.containsKey(TicketIEX.CITY_ARR))
        {
            m.put(ORIGIN, RASP);
        }
    }

    public void uniqueAllTheFlights() {
        if (ticketJson != null) {
            uniqFlights = new TicketFlightSet(this, ticketJson);
            ticketJson.clear();
            for (final TicketFlight x : uniqFlights.transferlessFlight()) {
                ticketJson.add(x.getJson());
            }
        }
    }

    public void mergeAllUniqFlightsInOne() {
        if (uniqFlights != null
            && uniqFlights.transferlessFlight().size() > 0)
        {
            List<TicketFlight> lf = uniqFlights.transferlessFlight();
            Object typeFromExtractor = null;
            Object url = null;
            TicketFlight fisrt = lf.get(0);
            for (TicketFlight flight: lf) {
                if (!TicketFlightSet.checkMinimumRequiredSet(
                    flight.getJson()))
                {
                    if (TicketWidgetType.BOOKING.equals(
                        flight.getJson().get(TYPE_FROM_EXTRACTOR)))
                    {
                        typeFromExtractor = TicketWidgetType.BOOKING;
                        url = flight.getJson().get(URL);
                    }
                }
            }
            lf.removeIf(f ->
                !TicketFlightSet.checkMinimumRequiredSet(f.getJson()));
            if (lf.isEmpty()) {
                lf.add(fisrt);
            }
            Map<String, Object> result = new HashMap<>(lf.get(0).getJson());
            String prefix;
            int id = 1;
            for (int i = 1; i < lf.size(); ++i) {
                if (lf.get(i).getArrCity().equals(lf.get(0).getDepCity())) {
                    prefix = BACK;
                } else {
                    if (i == lf.size() - 1) {
                        prefix = LAST;
                    } else {
                        prefix = '_' + String.valueOf(id++);
                    }
                }
                mergeOut(result, lf.get(i), prefix);
            }
            result.remove(TicketWidgetType.TYPE);
            if (typeFromExtractor != null) {
                result.put(TYPE_FROM_EXTRACTOR, typeFromExtractor);
            }
            if (url != null) {
                result.put(URL, url);
            }
            XJsonUtils.removeEntry(result, TicketWidgetType.TYPE);
            XJsonUtils.removeEntry(result, TicketWidgetType.TYPE + BACK);
            addTicketNumberToMergedInfo(result);
            ticketJson.add(0, result);
            ptrToMainTicketJson = result; // first element of ticketJson
        }
    }

    private void mergeOut(
        final Map<String, Object> r,
        final TicketFlight f,
        final String suffix)
    {
        for (final Map.Entry<String, Object> x : f.getJson().entrySet()) {
            r.put(x.getKey() + suffix, x.getValue());
        }
    }

    private void addTicketNumberToMergedInfo(final Map<String, Object> result) {
        if (ticketNumbersFromMicro.size() > 0) {
            putTicketNumberToJson(result, ticketNumbersFromMicro);
        } else {
            try {
                Object ticketNumbersObject =
                    XJsonUtils.getNodeByPathOrNull(
                        inputJson,
                        TICKET,
                        TICKET_UNDERLINE_NUMBER);
                if (ticketNumbersObject != null
                         && ticketNumbersObject instanceof List)
                {
                    List<?> ticketNumbers = (List<?>) ticketNumbersObject;
                    if (!ticketNumbers.isEmpty()) {
                        putTicketNumberToJson(
                            result,
                            new HashSet<>(ticketNumbers));
                    }
                }
            } catch (JsonUnexpectedTokenException e) {
                session()
                    .logger()
                    .log(
                        Level.INFO,
                        "Failed to get ticket_number from input json");
            }
        }
    }

    private void putTicketNumberToJson(
        final Map<String, Object> result,
        final Set<Object> ticketNumbers)
    {
        if (ticketNumbers.size() == 1) {
            result.put(
                TICKET_UNDERLINE_NUMBER,
                ticketNumbers.iterator().next());
        } else {
            result.put(TICKET_UNDERLINE_NUMBER, ticketNumbers);
        }
    }

    public TicketFlightSet uniqFlights() {
        return uniqFlights;
    }

    public void preParseAction(final Map<?, ?> inputJson)
        throws JsonUnexpectedTokenException, BadRequestException
    {
        mid = session.params().getString("mid");
        uid = session.params().getString("uid");
        receivedDate =
            ValueUtils.asLong(session.params().getString("received_date"));
        email = ValueUtils.asString(session.params().getString(USER_EMAIL));
        if (inputJson.containsKey(DOMAIN)) {
            domain =
                ValueUtils.asString(session.params().getString(DOMAIN));
        }
        subject = session.params().getString("subject", subject);
        if (domain.isEmpty()) {
            from = ValueUtils.asString(session.params().getString(EMAIL));
            domain = IndexationContext.extractDomain(from);
        }
    }

    public void updateWebcheckinUrl(final Map<String, Object> x) {
        String scheme = "http://";
        if (!x.containsKey(CHECKIN_URL)) {
            String iata = "";
            if (x.containsKey(TicketIEX.AIRLINE_IATA)) {
                iata =
                    XJsonUtils.getStrValue(x, TicketIEX.AIRLINE_IATA);
            }
            String url = TicketDomainParser.getInstance()
                .getCheckinUrl(domain, iata);
            if (url != null && !url.isEmpty()) {
                x.put(CHECKIN_URL, url);
            }
        } else {
            String url = XJsonUtils.getStrValue(x, CHECKIN_URL);
            if (!url.isEmpty()) {
                x.put(CHECKIN_URL, XStrAlgo.completeUrl(url, scheme));
            }
        }
        if (!x.containsKey(CHECKIN_URL + BACK)) {
            String iata = "";
            if (x.containsKey(TicketIEX.AIRLINE_IATA_BACK)) {
                iata =
                    XJsonUtils.
                        getStrValue(x, TicketIEX.AIRLINE_IATA_BACK);
            }
            String url = TicketDomainParser.getInstance()
                .getCheckinUrl(domain, iata);
            if (url != null && !url.isEmpty()) {
                x.put(CHECKIN_URL + BACK, url);
            }
        } else {
            String url = XJsonUtils.getStrValue(x, CHECKIN_URL + BACK);
            if (!url.isEmpty()) {
                x.put(
                    CHECKIN_URL + BACK, XStrAlgo.completeUrl(url, scheme));
            }
        }
    }

    public void postParseAction() {
        if (ticketJson != null) {
            boolean taksa = true;
            for (final Map<String, Object> x : ticketJson) {
                //Time to "MM.dd.yyy hh.mm.ss" format
                formatDateAndTime(x, "cancellation_info");
                long flightTime = XTimeUtils.getTimestampFromDateAndTime(
                    x,
                    TicketIEX.DATE_DEP,
                    TicketIEX.TIME_DEP);
                dataVerifyerIexMsgAdder(x, flightTime);
                //htmlSanitizer.mapSanitizer(x);
                formatTimesForTaksa(x, taksa);
                defineWidgetSubtype(x, flightTime);
            }
        }
        if (inputJson != null) {
            Object micro =
                XJsonUtils.getNodeByPathOrNullEless(inputJson, MICRO);
            Object unkonwn = XJsonUtils.getNodeByPathOrNullEless(
                micro,
                UNKNOWN);
            if (unkonwn == null) {
                micro = null;
            }
            if (micro == null) {
                micro =
                    XJsonUtils.getNodeByPathOrNullEless(
                        inputJson,
                        MICROHTML);
                unkonwn = XJsonUtils.getNodeByPathOrNullEless(
                    micro,
                    UNKNOWN);
                if (unkonwn == null) {
                    micro = null;
                }
            }
            if (micro != null) {
                Map<String, Object> rawdata = new HashMap<>();
                rawdata.put("rawdata", micro);
                ticketJson.add(rawdata);
            }
        }
        addAttachmentPartsToMainJson();
    }

    public void getTransferType() {
        if (ticketJson != null) {
            for (final Map<String, Object> x : ticketJson) {
                String transfer = null;
                String city = XJsonUtils.getStrValue(x, TicketIEX.CITY_DEP);
                String time = XJsonUtils.getStrValue(x, TicketIEX.TIME_DEP);
                if (city != null && time != null) {
                    transfer = TicketTransfer.getInstance()
                        .getTransferType(city, time, true);
                }
                if (transfer == null) {
                    city = XJsonUtils.getStrValue(x, TicketIEX.CITY_ARR);
                    time = XJsonUtils.getStrValue(x, TicketIEX.TIME_ARR);
                    if (city != null && time != null) {
                        transfer = TicketTransfer.getInstance().getTransferType(
                            city,
                            time,
                            false);
                    }
                }
                if (transfer != null && !transfer.isEmpty()) {
                    x.put(TRANSFER, transfer);
                }
            }
        }
    }

    public String getSubject() {
        return subject;
    }

    private void setBookingToday(
        final Map<String, Object> x,
        final long flightTime)
    {
        if (flightTime >= receivedDate
            && flightTime - receivedDate
            <= XTimeUtils.dayInSeconds()
            && !domain.endsWith("blablacar.ru")
            && !getTypes().contains(MessageType.REMINDER))
        {
            x.put(TicketWidgetType.TYPE, BOOKING_TODAY);
        }
    }

    public void defineWidgetSubtype(
        final Map<String, Object> x,
        final long flightTime)
    {
        if (flightTime != -1) {
            setBookingToday(x, flightTime);
            x.put("date_dep_ts", String.valueOf(flightTime));
        }
        String curType =
            XJsonUtils.getStrValue(x, TicketWidgetType.TYPE);
        if (curType.equals(TicketWidgetType.FLIGHT)) {
            return;
        }
        String widgetSubtype = TicketWidgetType.ETICKET;
        String origin = XJsonUtils.getStrValue(x, ORIGIN);
        boolean bookingToday = curType.equals(BOOKING_TODAY);
        if (bookingToday
            || curType.isEmpty()
            || curType.equals(UNDEFINED))
        {
            if (isFlightStatusChanged(x)) {
                widgetSubtype = TicketWidgetType.CHANGING;
            } else if (isFlightCancalled(x)
                || getTypes().contains(MessageType.CANCEL))
            {
                widgetSubtype = TicketWidgetType.CANCELING;
            } else if (getTypes().contains(MessageType.REMINDER)
                || domain.contains("yandex"))
            {
                // yandex flight tickets are reminder only
                widgetSubtype = REMINDER;
            } else if (origin.equals(REGEXP_P + PDF)
                || origin.equals(REGEXP_P + IEX_PATTERNS)
                || origin.equals(REGEXP_P + HTML))
            {
                if (x.containsKey(URL)) {
                    x.put(TicketWidgetType.TYPE, "eticket_link");
                } else {
                    widgetSubtype = UNDEFINED;
                }
            }
            if (bookingStatus) {
                widgetSubtype = TicketWidgetType.BOOKING;
            }
            if (bookingToday
                && (widgetSubtype.equals(TicketWidgetType.ETICKET)
                    || widgetSubtype.equals(TicketWidgetType.BOOKING)))
            {
                widgetSubtype = BOOKING_TODAY;
            }
            x.put(TicketWidgetType.TYPE, widgetSubtype);
        }
        Object typeFromExtractor = x.get(TYPE_FROM_EXTRACTOR);
        if (typeFromExtractor != null && !typeFromExtractor.equals(UNDEFINED)) {
            x.put(TicketWidgetType.TYPE, typeFromExtractor);
        }
    }

    private void formatTimesForTaksa(
        final Map<String, Object> x,
        final boolean taksaFlag)
    {
        //TODO: this need only for first taksa flight
        String postfix = "";
        for (int i = 0; i <= ticketJson.size() && (taksaFlag || i == 0); ++i) {
            if (i != 0) {
                postfix = '_' + String.valueOf(i);
            }
            if (i == ticketJson.size()) {
                postfix = LAST;
            }
            for (final String time : times) {
                XTimeUtils.changeTimeFormat(x, time + postfix);
            }
            completeDateAndTime(
                x,
                new String[]{
                    TicketIEX.DATE_DEP + postfix,
                    TicketIEX.TIME_DEP + postfix,
                    TicketIEX.DATE_DEP_TZ + postfix});
            completeDateAndTime(
                x,
                new String[]{
                    TicketIEX.DATE_ARR + postfix,
                    TicketIEX.TIME_ARR + postfix,
                    TicketIEX.DATE_ARR_TZ + postfix});
            completeDateAndTime(
                x,
                new String[]{
                    TicketIEX.DATE_DEP_BACK + postfix,
                    TicketIEX.TIME_DEP_BACK + postfix,
                    TicketIEX.DATE_DEP_TZ + BACK + postfix});
            completeDateAndTime(
                x,
                new String[]{
                    TicketIEX.DATE_ARR_BACK + postfix,
                    TicketIEX.TIME_ARR_BACK + postfix,
                    TicketIEX.DATE_ARR_TZ + BACK + postfix});
        }
    }

    private void formatDateAndTime(
        final Map<String, Object> x,
        final String keyToFormat)
    {
        if (x.containsKey(keyToFormat)) {
            String valueToFormat = XJsonUtils.getStrValue(x, keyToFormat);
            if (!valueToFormat.isEmpty()) {
                String date = XTimeUtils.formatDate(valueToFormat);
                String time = XTimeUtils.formatTime(valueToFormat);
                if (!date.isEmpty() && !time.isEmpty()) {
                    x.put(keyToFormat, date + ' ' + time);
                }
            }
        }
    }

    private void completeCityDepAndArray(
        final Map<String, Object> x,
        final String airportType,
        final String prefix)
    {
        final String key = "city" + prefix;
        if (!x.containsKey(key)) {
            String city = null;
            if (x.containsKey(airportType + IATA)) {
                city = TicketAirportParser.getInstance()
                    .iata2city((String) x.get(airportType + IATA));
            }
            if (city != null && x.containsKey(airportType)) {
                Object airport = x.get(airportType);
                if (domain.equals("onetwotrip.com")) {
                    city = (String) airport;
                } else {
                    if (airport instanceof String) {
                        String[] ms = ((String) airport).split(SPACE_AS_STR);
                        if (ms.length > 1) {
                            city = ms[0];
                        }
                    }
                }
            }
            if (city != null && !city.isEmpty()) {
                x.put(key, city);
            }
        }
    }

    private void dataVerifyerIexMsgAdder(
        final Map<String, Object> x,
        final long depDate)
    {
        // https://st.yandex-team.ru/IEX-1262
        if (depDate < receivedDate) {
            XJsonUtils.addStr(x, IEX_MSG, "date_dep ("
                + depDate + ") < received_date " + receivedDate);
        }
        String origin = XJsonUtils.getStrValueOrEmpty(x, ORIGIN);
        String url = XJsonUtils.getStrValueOrEmpty(x, URL);
        if (origin.equals(REGEXP_P + PDF) || origin.equals(REGEXP_P + HTML)) {
            XJsonUtils.addStr(x, IEX_MSG, "not found in rasp");
        }
        if (x.containsKey(REGEXP_EXTRACTION)) {
            Object regexpExtr = x.get(REGEXP_EXTRACTION);
            if (regexpExtr instanceof Map) {
                try {
                    long regexpDepDate =
                        XTimeUtils.getTimestampFromDateAndTimeMask(
                            ValueUtils.asMap(regexpExtr),
                            TicketIEX.DATE_DEP,
                            TicketIEX.DATE_DEP);
                    if (regexpDepDate != depDate) {
                        XJsonUtils.addStr(
                            x,
                            IEX_MSG,
                            "not match rasp");
                    }
                } catch (JsonUnexpectedTokenException e) {
                }
            }
        }
        if (depDate < receivedDate
            - XTimeUtils.dayInSeconds() * XTimeUtils.DAYS_IN_YEAR
            * XTimeUtils.TEN_YEARS)
        {
            XJsonUtils.addStr(x, IEX_MSG, "invalid date_dep (past): "
                + depDate);
            if (url.isEmpty()) {
                x.put(TicketWidgetType.TYPE, TicketWidgetType.UNDEFINED);
            }
        }
        if (depDate > receivedDate
            + XTimeUtils.dayInSeconds() * XTimeUtils.DAYS_IN_YEAR)
        {
            XJsonUtils.addStr(x, IEX_MSG, "invalid date_dep (future): "
                + depDate);
            if (url.isEmpty()) {
                x.put(TicketWidgetType.TYPE, TicketWidgetType.UNDEFINED);
            }
        }
        if (!formatFlightNumber(
            x,
            TicketIEX.FLIGHT_NUMBER,
            TicketIEX.AIRLINE_IATA))
        {
            XJsonUtils.addStr(x, IEX_MSG, "invalid or absent flight_number");
            if (url.isEmpty()) {
                x.put(TicketWidgetType.TYPE, TicketWidgetType.UNDEFINED);
            }
        }
        if (!formatFlightNumber(
            x,
            TicketIEX.FLIGHT_NUMBER_BACK,
            TicketIEX.AIRLINE_IATA_BACK))
        {
            XJsonUtils.addStr(
                x,
                IEX_MSG,
                "invalid or absent flight_number_back");
        }
        if (depDate < receivedDate
            && (x.containsKey(RASP_HTML_ORIGIN_ID)
            || x.containsKey(RASP_PDF_ORIGIN_ID)))
        {
            XJsonUtils.addStr(x, IEX_MSG, "rasp unreliable - old data");
        }
    }

    private boolean formatFlightNumber(
        final Map<String, Object> x,
        final String key,
        final String airline)
    {
        try {
            if (x.containsKey(key)) {
                String flightNumber = XJsonUtils.getStrValue(x, key);
                if (!flightNumber.isEmpty()) {
                    String result = flightNumber;
                    ArrayList<String> res =
                        XRegexpUtils.getFlightAsParts(flightNumber);
                    if (res.size() == 2) {
                        result = XStrAlgo.replaceRustoEng(res.get(0));
                        result += ' ';
                        result += Integer.parseInt(res.get(1));
                    } else {
                        if (x.containsKey(airline)) {
                            String firstPart = XJsonUtils.
                                getStrValue(x, airline);
                            if (!firstPart.isEmpty()) {
                                result = XStrAlgo.replaceRustoEng(firstPart);
                                result += ' ';
                                result +=
                                    Integer.parseInt(flightNumber);
                            }
                        }
                    }
                    x.put(key, result);
                    return true;
                }
            }
        } catch (NumberFormatException e) {
        }
        return false;
    }

    private void addAttachmentPartsToMainJson() {
        if (pdfParts != null && ptrToMainTicketJson != null) {
            ptrToMainTicketJson.put("print_parts", pdfParts);
        }
    }

    private boolean parseTicketInfo(final Map<?, ?> json)
        throws JsonUnexpectedTokenException
    {
        boolean microStatus = false;
        boolean commonStatus = false;
        boolean patternsStatus = false;
        for (TicketUnit x : ticketsUnits) {
            x.parse(json, microStatus || patternsStatus);
            if (x.getOrigin().startsWith(MICRO)) {
                microStatus |= x.getStatus();
            }
            if (x.getOrigin().equals(IEX_PATTERNS)) {
                patternsStatus |= x.getStatus();
            }
            commonStatus |= x.getStatus();
        }
        return commonStatus;
    }

    private boolean isFlightStatusChanged(final Map<String, Object> x) {
        return XStrAlgo.isMapValueContains(x, CHANGE_STATUS, "измен");
    }

    private boolean isFlightCancalled(final Map<String, Object> x) {
        //TODO: make config for words
        return XStrAlgo.isMapValueContains(x, CHANGE_STATUS, "отмен")
            || XStrAlgo.isMapValueContains(x, CHANGE_STATUS, "аннуляция")
            || XStrAlgo.isMapValueContains(x, CHANGE_STATUS, "возвра")
            || (x.containsKey(TicketAeroflot.RESERVATIONSTATUS)
            && x.get(TicketAeroflot.RESERVATIONSTATUS)
            .equals("http://schema.org/Cancelled"));
    }

    /*private boolean isFlightEligible(final Map<String, Object> x) {
        return x.containsKey(TICKET_NUMBER) // aeroflot sign
            || (x.containsKey(TicketAeroflot.RESERVATIONSTATUS)
            && x.get(TicketAeroflot.RESERVATIONSTATUS) // s7 sign
            .equals("http://schema.org/ReservationConfirmed"))
            || (x.containsKey(ORIGIN) && x.get(ORIGIN).equals(RASP))
            || (x.containsKey(ORIGIN) && x.get(ORIGIN).equals(PDF))
            || getDomain().equals("sindbad.ru")
            || (x.containsKey(FLIGHT_TYPE)
            && x.get(FLIGHT_TYPE).equals("FlightReservation"));
    }*/

    public ArrayList<String> getRaspData() {
        return raspData;
    }

    @Override
    public void response() {
        if (isCalled) {
            XMessageToLog.warning(this, "Response recall has happened.");
        } else {
            postParseAction();
        }
        session().response(
            YandexHttpStatus.SC_OK,
            JsonType.NORMAL.toString(ticketJson));
        isCalled = true;
    }

    public void failed(final Exception e) {
        session()
            .logger()
            .log(Level.WARNING, "Failed to process: "
                + humanReadableJson()
                + '\n' + session().listener().details()
                + " because of exception", e);
        session().handleException(
            HttpExceptionConverter.toHttpException(e));
    }

    private Map<?, ?> asMapOrNullWithCheck(
        final Map<?, ?> json,
        final String key)
        throws JsonUnexpectedTokenException
    {
        if (json != null && json.containsKey(key)) {
            Object res = json.get(key);
            if (res instanceof Map) {
                return (Map<?, ?>) res;
            }
        }
        return null;
    }

    private List<?> asListOrNullWithCheck(
        final Map<?, ?> json,
        final String key)
        throws JsonUnexpectedTokenException
    {
        if (json != null && json.containsKey(key)) {
            Object res = json.get(key);
            if (res instanceof List) {
                return (List<?>) res;
            }
        }
        return null;
    }

    private boolean parseIexFlightAndDate(
        final Map<?, ?> json,
        final String origin)
        throws JsonUnexpectedTokenException
    {
        List<?> flightNumAndDate = null;
        Object tryRegexpList =
            XJsonUtils.getNodeByPathOrNull(json, TICKET, "regexp");
        if (tryRegexpList instanceof List) {
            flightNumAndDate = ValueUtils.asList(tryRegexpList);
        }
        if (flightNumAndDate == null) {
            flightNumAndDate = asListOrNullWithCheck(json, TICKET);
        }
        if (flightNumAndDate != null) {
            for (final Object x : flightNumAndDate) {
                Map<?, ?> dn = ValueUtils.asMapOrNull(x);
                if (dn != null
                    && dn.size() == 2
                    && dn.containsKey(DATE)
                    && dn.containsKey(NUMBER))
                {
                    String date = ValueUtils.asString(dn.get(DATE));
                    String numr = ValueUtils.asString(dn.get(NUMBER));
                    numr = numr.replaceAll("\\s+", "");
                    ArrayList<String> res =
                        XRegexpUtils.getFlightAsParts(numr);
                    if (res.size() == 2) {
                        numr = res.get(0);
                        numr += Integer.parseInt(res.get(1));
                    }
                    if (checkDateAndNumberFormat(date, numr)) {
                        raspData.add(
                            numr + "&date=" + date + "&origin=" + origin);
                    }
                } else if (dn != null
                    && dn.size() == 1
                    && dn.containsKey("booking_status"))
                {
                    bookingStatus = true;
                }
            }
        }
        if (raspData.size() > 0) {
            return true;
        }
        return false;
    }

    private boolean checkDateAndNumberFormat(final String d, final String n) {
        return TIME_DATE_RASP_PATTERN.matcher(d).matches() && !n.isEmpty();
    }

    private boolean parsePdfData(
        final Map<?, ?> flight,
        final boolean microStatus,
        final String origin)
        throws JsonUnexpectedTokenException
    {
        boolean res = false;
        List<String> pdfparts = new LinkedList<>();
        if (flight.containsKey(PDF)) {
            Map<?, ?> tmpMp = ValueUtils.asMap(flight.get(PDF));
            if (!microStatus) {
                res =
                    parseIexFlightAndDate(tmpMp, origin);
                if (!res) {
                    res =
                        parseIexPatternsData(
                            asMapOrNullWithCheck(tmpMp, TICKET), PDF);
                }
            }
            if (tmpMp.containsKey(HID + PDF)) {
                pdfparts.add(ValueUtils.asString(tmpMp.get(HID + PDF)));
            }
        }
        for (int i = 1; flight.containsKey(PDF + i); i++) {
            Map<?, ?> tmpMp = ValueUtils.asMap(flight.get(PDF + i));
            if (!microStatus) {
                boolean tmpRes =
                    parseIexFlightAndDate(tmpMp, origin);
                if (!tmpRes) {
                    tmpRes =
                        parseIexPatternsData(
                            asMapOrNullWithCheck(tmpMp, TICKET), PDF + i);
                }
                res |= tmpRes;
            }
            if (tmpMp.containsKey(HID + PDF)) {
                pdfparts.add(ValueUtils.asString(tmpMp.get(HID + PDF)));
            }
        }
        pdfParts = pdfparts;
        return res;
    }

    private boolean parseIexPatternsData(
        final Map<?, ?> flight,
        final String origin)
        throws JsonUnexpectedTokenException
    {
        if (flight != null) {
            List<?> l = asListOrNullWithCheck(flight, "flight");
            if (l != null) {
                for (final Object x : l) {
                    HashMap<String, Object> subTicketJson = new HashMap<>();
                    Map<?, ?> fm = ValueUtils.asMapOrNull(x);
                    if (fm != null) {
                        subTicketJson.put(ORIGIN, origin);
                        for (final Map.Entry<?, ?> x2 : fm.entrySet()) {
                            String key = ValueUtils.asString(x2.getKey());
                            subTicketJson.put(key, x2.getValue());
                            // for anywayanyday
                            if (PAYMENT_STATUS.equals(key)
                                && checkPaymentStatusUnpaid(x2.getValue()))
                            {
                                subTicketJson.put(
                                    TYPE_FROM_EXTRACTOR,
                                    TicketWidgetType.BOOKING);
                            }
                        }
                    }
                    ticketJson.add(subTicketJson);
                }
                return ticketJson.size() > 0;
            }
        }
        return false;
    }

    private boolean checkPaymentStatusUnpaid(final Object value) {
        String status = XJsonUtils.asStringOrNull(value);
        return status != null
            && PAYMENT_STATUS_UNPAID_PATTERN.matcher(status).matches();
    }

    private boolean parseMicroData(final Map<?, ?> micro, final String origin)
        throws JsonUnexpectedTokenException
    {
        List<?> l = asListOrNullWithCheck(micro, UNKNOWN);
        if (l == null) {
            // trip.ru json location
            Object res =
                XJsonUtils.getNodeByPathOrNull(micro, UNKNOWN, "reservation");
            if (res instanceof List) {
                l = (List) res;
            }
        }
        if (l == null) {
            Map<?, ?> tryMap = asMapOrNullWithCheck(micro, UNKNOWN);
            if (tryMap != null) {
                l = Arrays.asList(tryMap);
            }
        }
        int prevSize = ticketJson.size();
        if (l != null) {
            for (final Object x : l) {
                Map<?, ?> m = ValueUtils.asMapOrNull(x);
                saveTicketNumberFromMicroIfPresent(m);
                Object reservationFor =
                    XJsonUtils.getNodeByPathOrNullEless(
                        m,
                        TicketAeroflot.RESERVATIONFOR);
                if (reservationFor == null) {
                    reservationFor =
                    XJsonUtils.getNodeByPathOrNullEless(
                        m,
                        TicketAeroflot.RESERVATION,
                        TicketAeroflot.RESERVATIONFOR);
                }
                if (m != null && reservationFor instanceof Map) {
                    Map<?, ?> m1 =
                        ValueUtils.asMap(reservationFor);
                    Map<String, Object> root = new HashMap<>();
                    for (final Map.Entry<?, ?> xmr : m.entrySet()) {
                        Object kx = xmr.getKey();
                        Object vx = xmr.getValue();
                        if (kx instanceof String && vx instanceof String) {
                            root.put((String) kx, vx);
                        }
                    }
                    getFlatJson(m1, root, "");
                    fixFlightNumber(root);
                    int flightIndex = createNewFlightIndex();
                    for (final Map.Entry<String, Object> x1 : root.entrySet()) {
                        if (x1.getValue().equals("PostalAddress")) {
                            continue; // redundant field
                        }
                        pushToOutputJson(
                            flightIndex,
                            x1.getKey(),
                            x1.getValue());
                    }
                    pushToOutputJson(flightIndex, ORIGIN, origin);
                    pushToJsonAndCheck(
                        flightIndex,
                        m,
                        TicketAeroflot.RESERVATIONNUMBER);
                    pushToJsonAndCheck(flightIndex, m, TICKET_NUMBER);
                    pushToJsonAndCheck(flightIndex, m, "bookingTime");
                    pushToJsonAndCheck(flightIndex, m, FLIGHT_TYPE);
                    pushToJsonAndCheck(
                        flightIndex,
                        m,
                        TicketAeroflot.RESERVATIONSTATUS);
                    // Specific city extraction from an entry
                    // - "airport_dep": "Москва, Домодедово"
                    String url = XJsonUtils.getStrValue(m, URL);
                    if (url.contains("myb.s7.ru")
                        || getDomain().contains("utair.ru")
                        || getDomain().contains(JETSTAR_COM))
                    {
                        String cityDepAirport = XJsonUtils.getStrValue(
                            root,
                            TicketAeroflot.DEPARTUREAIRPORT
                            + TicketAeroflot.NAME);
                        String[] res = cityDepAirport.split(COMMA);
                        if (res.length == 1) {
                            pushToOutputJson(
                                flightIndex,
                                TicketIEX.CITY_DEP,
                                res[0]);
                        }
                        if (res.length == 2) {
                            pushToOutputJson(
                                flightIndex,
                                TicketIEX.CITY_DEP,
                                res[0]);
                            pushToOutputJson(
                                flightIndex,
                                TicketAeroflot.DEPARTUREAIRPORT
                                    + TicketAeroflot.NAME,
                                res[1]);
                        }
                        String cityArrAirport = XJsonUtils.getStrValue(
                            root,
                            TicketAeroflot.ARRIVALAIRPORT
                             + TicketAeroflot.NAME);
                        res = cityArrAirport.split(COMMA);
                        if (res.length == 1) {
                            pushToOutputJson(
                                flightIndex,
                                TicketIEX.CITY_ARR,
                                res[0]);
                        }
                        if (res.length == 2) {
                            pushToOutputJson(
                                flightIndex,
                                TicketIEX.CITY_ARR,
                                res[0]);
                            pushToOutputJson(
                                flightIndex,
                                TicketAeroflot.ARRIVALAIRPORT
                                    + TicketAeroflot.NAME,
                                res[1]);
                        }
                    }
                    completeCityDepAndArray(
                        ticketJson.get(flightIndex),
                        TicketIEX.AIRPORT_ARR,
                        "_arr");
                    completeCityDepAndArray(
                        ticketJson.get(flightIndex),
                        TicketIEX.AIRPORT_DEP,
                        "_dep");
                }
            }
            return prevSize < ticketJson.size();
        }
        return false;
    }

    private void saveTicketNumberFromMicroIfPresent(
        final Map<?, ?> ticketInfoSubJson)
    {
        if (ticketInfoSubJson != null
            && ticketInfoSubJson.containsKey(TICKET_NUMBER))
        {
            Object ticketNumber = ticketInfoSubJson.get(TICKET_NUMBER);
            if (ticketNumber != null && !((String) ticketNumber).isEmpty()) {
                ticketNumbersFromMicro.add(ticketNumber);
            }
        }
    }

    private void getFlatJson(
        final Map<?, ?> source,
        final Map<String, Object> dist,
        final String prefix)
    {
        for (final Map.Entry<?, ?> x : source.entrySet()) {
            Object key = x.getKey();
            Object value = x.getValue();
            if (key instanceof String) {
                if (value instanceof String || value instanceof Number) {
                    dist.put(prefix + key, value.toString());
                } else if (value instanceof Map) {
                    getFlatJson((Map<?, ?>) value, dist, prefix + key);
                }
            }
        }
    }

    private int createNewFlightIndex() {
        int flightIndex = ticketJson.size();
        ticketJson.add(new HashMap<String, Object>());
        return flightIndex;
    }

    private void pushToJsonAndCheck(
        final int flightIndex,
        final Map<?, ?> m,
        final String key)
    {
        if (m.containsKey(key)) {
            pushToOutputJson(flightIndex, key, m.get(key));
        }
    }

    private void pushToOutputJson(
        final int flightIndex,
        final Object key,
        final Object value)
    {
        Map<String, Object> m;
        Object mutableKey = key;
        if (MICRO2IEX.containsKey(mutableKey)) {
            //rename IEX-pattern name to IEX-ticket name
            mutableKey = MICRO2IEX.get(mutableKey);
        }
        m = ticketJson.get(flightIndex);
        if (mutableKey instanceof String) {
            String keyr = (String) mutableKey;
            m.put(keyr, value.toString().trim());
        } else if (mutableKey instanceof ArrayList) {
            for (Object x : (ArrayList<?>) mutableKey) {
                if (x instanceof String) {
                    String keyr = (String) x;
                    m.put(keyr, value.toString().trim());
                }
            }
        }
    }

    private void fixFlightNumber(final Map<String, Object> root) {
        //sindbad.ru doesn't have full fligthnumber
        String fn = (String) root.get(TicketAeroflot.FLIGHTNUMBER);
        if (fn != null) {
            String iata = (String) root.get("provideriataCode");
            if (iata == null) {
                iata = (String) root.get("airlineiataCode");
            }
            if (iata != null && !fn.startsWith(iata)) {
                root.put(TicketAeroflot.FLIGHTNUMBER, iata + fn);
            }
        }
    }
}
