package ru.yandex.iex.proxy.tickethandlerlegacy;

import java.io.UnsupportedEncodingException;
import java.security.NoSuchAlgorithmException;
import java.text.ParseException;
import java.util.Map;

import ru.yandex.iex.proxy.XJsonUtils;
import ru.yandex.iex.proxy.XMessageToLog;
import ru.yandex.iex.proxy.XRegexpUtils;
import ru.yandex.iex.proxy.XTimeUtils;
import ru.yandex.iex.proxy.XTools;
import ru.yandex.json.xpath.JsonUnexpectedTokenException;
import ru.yandex.json.xpath.ValueUtils;

class TicketFlight {
    private TicketContext context;
    private long depTimestamp = -1;
    private long arrTimestamp = -1;
    private Map<String, Object> ptrToFlight;
    private String ticketIdTimestamp = "";
    private String ticketIdISOTime = "";
    private String ticketId = "";

    TicketFlight(final TicketContext context, final Map<String, Object> fl) {
        ptrToFlight = fl;
        this.context = context;
        initTicketId();
        depTimestamp =
            initTimestamp(fl, TicketIEX.DATE_DEP, TicketIEX.TIME_DEP);
        arrTimestamp =
            initTimestamp(fl, TicketIEX.DATE_ARR, TicketIEX.TIME_ARR);
        ptrToFlight.put(TicketWidgetType.TYPE, TicketWidgetType.FLIGHT);
    }

    public String getTicketId() {
        return ticketId;
    }

    public String getFlightType() {
        try {
            if (ptrToFlight.containsKey(TicketWidgetType.TYPE)) {
                return ValueUtils.asString(
                    ptrToFlight.get(TicketWidgetType.TYPE));
            }
        } catch (JsonUnexpectedTokenException e) {
        }
        return TicketWidgetType.UNDEFINED;
    }

    public Map<String, Object> getJson() {
        return ptrToFlight;
    }

    public boolean isEqual(final TicketFlight b) {
        return ptrToFlight == b.ptrToFlight;
    }

    public boolean isDateLessThan(final TicketFlight b) {
        return depTimestamp < b.depTimestamp;
    }

    public boolean isValid() {
        return depTimestamp != -1
            && ptrToFlight != null;
    }

    private void initTicketId() {
        final String time = getTimeDep();
        final String date = getDateDep();
        String flight = getFlight();
        final String fullTime =
            XRegexpUtils.getDateInISO(date) + ' ' + XRegexpUtils.getTime(time);
        flight = flight.replaceAll("[- ]", "");
        try {
            ticketIdTimestamp =
                XTimeUtils.getDateTimestampMillisec(fullTime)
                    + ':' + flight;
            try {
                ticketId = XTools.slowMd5hex(ticketIdTimestamp);
            } catch (NoSuchAlgorithmException
                | UnsupportedEncodingException e)
            {
                ticketId = "error";
            }
            ticketIdISOTime = fullTime;
        } catch (ParseException e) {
            // It's not critical exception because
            // we don't need necessary set a date to the reminder
            XMessageToLog.warning(
                context,
                "Wrong when parsing date_dep and time_dep, time: "
                    + fullTime + " and flight: " + flight);
        }
    }

    private long initTimestamp(
        final Map<String, Object> fl,
        final String date,
        final String time)
    {
        long result =
            XTimeUtils.getTimestampFromDateAndTime(fl, date, time);
        // gmt is taken into account for accurate comparsions of times
        if (result == -1) {
            String itime = XRegexpUtils.getTime(
                XJsonUtils.getStrValue(fl, time));
            String idate = XRegexpUtils.getDateInISO(
                XJsonUtils.getStrValue(fl, date));
            XMessageToLog.warning(context, "Timestamp was not received, "
                + "check format of " + time + " and " + date + " in iex json."
                + " time: " + itime + " date: " + idate);
        }
        return result;
    }

    public String getFlight() {
        return XJsonUtils.getStrValue(ptrToFlight, TicketIEX.FLIGHT_NUMBER);
    }

    public String getTimeDep() {
        String time;
        time = XJsonUtils.getStrValue(ptrToFlight, TicketIEX.TIME_DEP);
        time = XRegexpUtils.timeToReminderTime(time);
        return time;
    }

    public String getDateDep() {
        String date;
        date = XJsonUtils.getStrValue(ptrToFlight, TicketIEX.DATE_DEP);
        date = XRegexpUtils.timeToReminderTime(date);
        return date;
    }

    public String getDepCity() {
        return XJsonUtils.getStrValue(ptrToFlight, TicketIEX.CITY_DEP);
    }

    public String getArrCity() {
        return XJsonUtils.getStrValue(ptrToFlight, TicketIEX.CITY_ARR);
    }

    public String getTicketIdTimestamp() {
        return ticketIdTimestamp;
    }

    public String getTicketIdISOTime() {
        return ticketIdISOTime;
    }

    public long getDepTimeStamp() {
        return depTimestamp;
    }

    public long getArrTimestamp() {
        return arrTimestamp;
    }

    @Override
    public String toString() {
        return ptrToFlight.toString();
    }
}
