package ru.yandex.iex.proxy.tickethandlerlegacy;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.TreeSet;

import ru.yandex.iex.proxy.XJsonUtils;
import ru.yandex.iex.proxy.XTimeUtils;

class TicketFlightSet {
    public static final String TICKET_ID = "ticketId";
    public static final String TRANSFER = "transfer";
    public static final String ARR = "arr";
    public static final String ORIGIN = "origin";
    public static final String BACK = "_back";
    public static final String LAST_BACK = "_last_back";
    public static final String LAST = "_last";

    private TreeSet<TicketFlight> allTicketFlights =
        new TreeSet<TicketFlight>(new FlightComparator());
    private TicketContext context;
    private List<TicketFlight> transferlessTicketFlights =
        new ArrayList<>();

    TicketFlightSet(
        final TicketContext context,
        final List<Map<String, Object>> listIexJson)
    {
        this.context = context;
        initAllFlights(listIexJson);
        initTranferlessFlights();
        initDebugTicketIdInfo();
    }

    public List<TicketFlight> transferlessFlight() {
        return transferlessTicketFlights;
    }

    public TreeSet<TicketFlight> allFlights() {
        return allTicketFlights;
    }

    private void initDebugTicketIdInfo() {
        for (final TicketFlight x : transferlessTicketFlights) {
            Map<String, Object> ticketIdEntry = new HashMap<>();
            ticketIdEntry.put(TICKET_ID + "_md5hex", x.getTicketId());
            ticketIdEntry.put(
                TICKET_ID + "_timestamp",
                x.getTicketIdTimestamp());
            ticketIdEntry.put(TICKET_ID + "_time", x.getTicketIdISOTime());
            x.getJson().put("debug_reminder", ticketIdEntry);
        }
    }

    private void initTranferlessFlights() {
        if (allTicketFlights.size() > 1) {
            List<TicketFlight> fl = new ArrayList<>(allTicketFlights);
            int l = 0;
            int r = 0;
            int end = fl.size();
            while (r < end) {
                while (r + 1 < end && isTransferCity(fl, r, r + 1)) {
                    r++;
                }
                transferlessTicketFlights.add(
                    mergeDepArrF(fl.get(l), fl.get(r)));
                r++;
                l = r;
            }
        } else if (allTicketFlights.size() == 1) {
            transferlessTicketFlights.add(allTicketFlights.first());
        }
    }

    private TicketFlight mergeDepArrF(
        final TicketFlight dep,
        final TicketFlight arr)
    {
        if (dep == arr) {
            return dep;
        }
        Map<String, Object> merge = new HashMap<>();
        for (final Map.Entry<String, Object> x : dep.getJson().entrySet()) {
            final String k = x.getKey();
            if (k.contains("dep") || !k.contains(ARR)) {
                merge.put(k, x.getValue());
            }
        }
        for (final Map.Entry<String, Object> x : arr.getJson().entrySet()) {
            final String k = x.getKey();
            if (k.contains(ARR)) {
                merge.put(k, x.getValue());
            }
        }
        if (!merge.containsKey(TRANSFER)
            && arr.getJson().containsKey(TRANSFER))
        {
            merge.put(TRANSFER, arr.getJson().get(TRANSFER));
        }
        return new TicketFlight(context, merge);
    }

    private boolean isTransferCity(
        final List<TicketFlight> l,
        final int p,
        final int n)
    {
        TicketFlight a = l.get(p);
        TicketFlight b = l.get(n);
        long timeInCity = a.getArrTimestamp();
        long timeOutCIty = b.getDepTimeStamp();
        return isTimeDifferByLessThanDay(timeInCity, timeOutCIty)
            && !isCycleFlight(a, b);
    }

    private boolean isTimeDifferByLessThanDay(
        final long timeA,
        final long timeB)
    {
        return Math.abs(timeB - timeA) < XTimeUtils.dayInSeconds();
    }

    private boolean isCycleFlight(final TicketFlight a, final TicketFlight b) {
        return a.getDepCity().equals(b.getArrCity());
    }

    private void initAllFlights(final List<Map<String, Object>> listIexJson) {
        if (listIexJson != null) {
            for (final Map<String, Object> x : listIexJson) {
                createNewFlightIfPossible(x);
                if (!XJsonUtils.isMapContainPair(x, ORIGIN, "micro")) {
                    Map<String, Object> lastBackFlight =
                        selectSubjsonWithSpecificPrefix(LAST_BACK, x);
                    createNewFlightIfPossible(lastBackFlight);
                    Map<String, Object> backFlight =
                        selectSubjsonWithSpecificPrefix(BACK, x);
                    createNewFlightIfPossible(backFlight);
                    Map<String, Object> lastFlight =
                        selectSubjsonWithSpecificPrefix(LAST, x);
                    createNewFlightIfPossible(lastFlight);
                }
            }
        }
    }

    private void createNewFlightIfPossible(
        final Map<String, Object> newFlight)
    {
        if (checkMinimumRequiredSet(newFlight)
            || newFlight.containsKey(TicketIEX.URL))
        {
            context.updateWebcheckinUrl(newFlight);
            TicketFlight t2 = new TicketFlight(context, newFlight);
            allTicketFlights.add(t2);
        }
    }

    public static boolean checkMinimumRequiredSet(
        final Map<String, Object> iexJs)
    {
        return iexJs.containsKey(TicketIEX.FLIGHT_NUMBER)
            && iexJs.containsKey(TicketIEX.TIME_DEP)
            && iexJs.containsKey(TicketIEX.DATE_DEP);
    }

    private Map<String, Object> selectSubjsonWithSpecificPrefix(
        final String prefix,
        final Map<String, Object> x)
    {
        // split out a new flight from iex-patterns result
        Map<String, Object> tmp = new HashMap<>();
        ArrayList<String> keysToRemove = new ArrayList<>();
        for (Map.Entry<String, Object> smplx : x.entrySet()) {
            final String key = smplx.getKey();
            if (key.contains(prefix)) {
                String backKey = key.replace(prefix, "");
                tmp.put(
                    backKey,
                    x.get(key));
                keysToRemove.add(key);
            }
        }
        for (final String key : keysToRemove) {
            x.remove(key);
        }
        XJsonUtils.copyEntry(ORIGIN, x, tmp);
        XJsonUtils.copyEntry(TicketWidgetType.TYPE, x, tmp);
        return tmp;
    }
}

class FlightComparator implements Serializable, Comparator<TicketFlight> {
    private static final long serialVersionUID = 6106269076155338045L;

    @Override
    public int compare(final TicketFlight o1, final TicketFlight o2) {
        return (int) (o1.getDepTimeStamp() - o2.getDepTimeStamp());
    }
}

