package ru.yandex.iex.proxy.tickethandlerlegacy;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;

import org.apache.http.HttpException;
import org.apache.http.HttpHost;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.BadResponseException;
import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.http.util.YandexHttpStatus;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.iex.proxy.AbstractEntityHandler;
import ru.yandex.iex.proxy.IexProxy;
import ru.yandex.iex.proxy.XJsonUtils;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.JsonBadCastException;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.xpath.JsonUnexpectedTokenException;
import ru.yandex.stater.RequestInfo;
import ru.yandex.util.timesource.TimeSource;

public class TicketHandler extends AbstractEntityHandler<TicketContext> {
    public static final String ORIGIN = "origin";
    private static final String DATE_DEP = "date_dep";
    private static final String ERROR = "error";

    public TicketHandler(final IexProxy iexProxy) {
        super(iexProxy, "ticket");
    }

    @Override
    protected TicketContext createContext(
            final IexProxy iexProxy,
            final ProxySession session,
            final Map<?, ?> json)
        throws HttpException, JsonUnexpectedTokenException
    {
        return new TicketContext(iexProxy, session, json);
    }

    @Override
    protected void handle(final TicketContext context)
        throws JsonUnexpectedTokenException, BadRequestException
    {
        if (context.getTicketJson() == null) {
            context.response();
            return;
        }
        if (context.getRaspData().isEmpty()) {
            // Probably we've got a result from micro or iex-patterns
            TicketHandler.callNextHandler(context);
            return;
        }
        // Goto to rasp service for the result
        AsyncClient client =
            context.iexProxy().raspClient().adjust(context.session().context());
        String request = "/v1/flight/mail/?number=";
        HttpHost host = context.iexProxy().config().raspConfig().host();
        try {
            final MultiFutureCallback<ActionWithResult> multiCallback =
                new MultiFutureCallback<>(
                    new FinishMulticallbacks(context));
            final List<ResultCallback> subCallbacks =
                new ArrayList<>(context.getRaspData().size());
            for (final String x : context.getRaspData()) {
                subCallbacks.add(
                    new ResultCallback(
                        context,
                        multiCallback.newCallback(),
                        x));
            }
            multiCallback.done();
            int i = 0;
            for (final String x : context.getRaspData()) {
                context.session().logger().log(
                    Level.INFO,
                    "Processing rasp request" + request + x);
                client.execute(
                    host,
                    new BasicAsyncRequestProducerGenerator(
                        request + x),
                    JsonAsyncTypesafeDomConsumerFactory.OK,
                    context.session().listener()
                        .createContextGeneratorFor(client),
                    subCallbacks.get(i++));
            }
        } catch (Exception e) {
            context.session().logger().log(
                Level.SEVERE,
                "GeoCoder request encoding failed",
                e);
            context.failed(e);
        }
    }

    private static class ActionWithResult {
        private final Map<String, Object> result;

        ActionWithResult(final Map<String, Object> result) {
            this.result = result;
        }

        public Map<String, Object> result() {
            return result;
        }
    }

    private static class ResultCallback
        implements FutureCallback<JsonObject>
    {
        private final long start = TimeSource.INSTANCE.currentTimeMillis();
        private final TicketContext context;
        private final FutureCallback<ActionWithResult> callback;
        private final String query;

        ResultCallback(
            final TicketContext context,
            final FutureCallback<ActionWithResult> callback,
            final String q)
        {
            this.context = context;
            this.query = "flight_number=" + q.replace("date", DATE_DEP);
            this.callback = callback;
        }

        private boolean isQuery() {
            return query != null && !query.isEmpty();
        }

        private Map<String, Object> exceptionToJson(final Exception e) {
            final StringBuilderWriter sbw = new StringBuilderWriter();
            e.printStackTrace(sbw);
            return Collections.singletonMap(ERROR, sbw.toString());
        }

        private Map<String, Object> getMapFromQuery() {
            HashMap<String, Object> result = new HashMap<>();
            if (isQuery()) {
                String[] param = query.split("&");
                for (String x : param) {
                    String[] value = x.split("=");
                    if (value.length == 2) {
                        result.put(value[0], value[1]);
                        if (value[0].equals(DATE_DEP)) {
                            result.put("time_dep", value[1]);
                        }
                    }
                }
            }
            return result;
        }

        private Map<String, Object> queryOrExceptionToJson(final Exception e) {
            if (isQuery()) {
                return getMapFromQuery();
            }
            return exceptionToJson(e);
        }

        @Override
        public void cancelled() {
            stat(YandexHttpStatus.SC_CLIENT_CLOSED_REQUEST);
            callback.cancelled();
        }

        @Override
        public void failed(final Exception e) {
            if (e instanceof BadResponseException) {
                int status = ((BadResponseException) e).statusCode();
                if (status == YandexHttpStatus.SC_NOT_FOUND) {
                    context.iexProxy().raspRequestCompleted(true);
                }
                stat(((BadResponseException) e).statusCode());
                context.session().logger().log(
                    Level.INFO,
                    "Completed rasp request, flight not found");
            } else {
                stat(YandexHttpStatus.SC_REMOTE_CLOSED_REQUEST);
            }
            callback.completed(
                new ActionWithResult(queryOrExceptionToJson(e)));
        }

        @Override
        public void completed(final JsonObject response) {
            context.iexProxy().raspRequestCompleted(false);
            Map<String, Object> queryMap = getMapFromQuery();
            try {
                JsonMap jsonMap = response.asMap();
                Map<String, Object> result = new HashMap<>();
                for (Map.Entry<String, JsonObject> entry: jsonMap.entrySet()) {
                    result.put(entry.getKey(), entry.getValue());
                }
                stat(YandexHttpStatus.SC_OK);
                if (queryMap.containsKey(ORIGIN)) {
                    String origin = "rasp_" + queryMap.get(ORIGIN);
                    result.put(ORIGIN, origin);
                    result.put(
                        "regexp_extraction",
                        queryMap);
                }
                callback.completed(new ActionWithResult(result));
            } catch (JsonBadCastException e) {
                stat(YandexHttpStatus.SC_SERVICE_UNAVAILABLE);
                callback.completed(
                    new ActionWithResult(queryOrExceptionToJson(e)));
            }
        }

        private void stat(final int status) {
            context.getRaspStater().accept(
                new RequestInfo(
                    TimeSource.INSTANCE.currentTimeMillis(),
                    status,
                    start,
                    start,
                    0L,
                    0L));
        }
    }

    private static class FinishMulticallbacks
        implements FutureCallback<List<ActionWithResult>>
    {
        private final TicketContext context;

        FinishMulticallbacks(final TicketContext context) {
            this.context = context;
        }

        @Override
        public void completed(final List<ActionWithResult> results) {
            for (final ActionWithResult x : results) {
                Map<String, Object> result = x.result();
                context.session().logger().log(
                    Level.INFO,
                    "Completed rasp request, response = "
                        + JsonType.NORMAL.toString(result));
                String origin = XJsonUtils.getStrValue(result, ORIGIN);
                context.renameRaspFields(result);
                if (!origin.isEmpty()) {
                    result.put(ORIGIN, origin);
                }
                context.getTicketJson().add(result);
            }
            TicketHandler.callNextHandler(context);
        }

        @Override
        public void cancelled() {
            context.response();
        }

        @Override
        public void failed(final Exception e) {
            context.failed(e);
        }
    }

    public static void callNextHandler(final TicketContext context) {
        context.uniqueAllTheFlights();
        //context.mergeAllUniqFlightsInOne();
//        context.defineAlldTheTypesAnsTransfers();
//        context.response();
        //return;
        //context.define AllTheTypes();
        new TicketGeoid(context);
        //(new TicketReminder(context)).handle();
        // - will be called in the TicketGeoid
    }
}

