package ru.yandex.iex.proxy.tickethandlerlegacy;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.nio.charset.StandardCharsets;
import java.util.HashSet;
import java.util.Locale;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.joda.time.DateTimeZone;
import org.joda.time.LocalTime;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;

public final class TicketTransfer {
    public static final String AEROEXPRESS = "aeroexpress";
    public static final String YTAXI = "ytaxi";
    public static final String TAXI = "taxi";

    private static TicketTransfer instance;

    private static final DateTimeFormatter FORMATTER =
        DateTimeFormat.forPattern("H:mm:ss")
            .withZone(DateTimeZone.forID("Europe/Moscow"));
    private static final Pattern TIME_PATTERN =
        Pattern.compile("[0-9]{1,2}:[0-9]{1,2}:[0-9]{1,2}");

    private static Set<String> ytaxiCities = new HashSet<>();
    private static LocalTime depStartTimeForAeroexpress;
    private static LocalTime depEndTimeForAeroexpress;
    private static LocalTime arrStartTimeForAeroexpress;
    private static LocalTime arrEndTimeForAeroexpress;

    static {
        depStartTimeForAeroexpress = FORMATTER.parseLocalTime("08:01:00");
        depEndTimeForAeroexpress = FORMATTER.parseLocalTime("01:59:00");
        arrStartTimeForAeroexpress = FORMATTER.parseLocalTime("05:01:00");
        arrEndTimeForAeroexpress = FORMATTER.parseLocalTime("22:59:00");
        try {
            instance = new TicketTransfer();
        } catch (IOException e) {
            throw new RuntimeException("TicketTransfer creation failed", e);
        }
    }

    private TicketTransfer() throws IOException {
        try (BufferedReader reader = new BufferedReader(
            new InputStreamReader(
                this.getClass().getResourceAsStream("ytaxi_cities.txt"),
                StandardCharsets.UTF_8));
            BufferedReader readerEn = new BufferedReader(
            new InputStreamReader(
                this.getClass().getResourceAsStream("ytaxi_cities_en.txt"),
                StandardCharsets.UTF_8)))
        {
            String city;
            while ((city = reader.readLine()) != null) {
                ytaxiCities.add(city.toLowerCase(Locale.getDefault()));
            }
            while ((city = readerEn.readLine()) != null) {
                ytaxiCities.add(city.toLowerCase(Locale.getDefault()));
            }
        }
    }

    public static TicketTransfer getInstance() {
        return instance;
    }

    public String getTransferType(
        final String city,
        final String time,
        final boolean isDeparture)
    {
        if (city == null || city.isEmpty()) {
            return null;
        }
        String type = null;
        if (isMoscow(city) && time != null && !time.isEmpty()) {
            Matcher matcher = TIME_PATTERN.matcher(time);
            try {
                if (matcher.find()) {
                    LocalTime parsedTime =
                        FORMATTER.parseLocalTime(matcher.group(0));
                    if (isAeroexpressTime(parsedTime, isDeparture)) {
                        type = AEROEXPRESS;
                    }
                }
            } catch (IllegalStateException ignored) {
            }
        }
        if (type == null) {
            if (isYandexTaxiCity(city)) {
                type = YTAXI;
            } else {
                type = TAXI;
            }
        }
        return type;
    }

    private boolean isMoscow(final String city) {
        String lowerCity = city.toLowerCase(Locale.getDefault());
        return lowerCity.equals("москва") || lowerCity.equals("moscow");
    }

    private boolean isAeroexpressTime(
        final LocalTime time,
        final boolean isDeparture)
    {
        LocalTime start;
        LocalTime end;
        if (isDeparture) {
            start = depStartTimeForAeroexpress;
            end = depEndTimeForAeroexpress;
        } else {
            start = arrStartTimeForAeroexpress;
            end = arrEndTimeForAeroexpress;
        }
        if (start.isBefore(end)) {
            return time.isAfter(start) && time.isBefore(end);
        } else {
            return !(time.isAfter(end) && time.isBefore(start));
        }
    }

    private boolean isYandexTaxiCity(final String city) {
        return ytaxiCities.contains(city.toLowerCase(Locale.getDefault()));
    }
}
