package ru.yandex.iex.proxy.xiva;

import java.io.IOException;
import java.net.URISyntaxException;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.TimerTask;
import java.util.logging.Level;

import org.apache.http.HttpHost;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.ContentType;
import org.apache.http.message.BasicHeader;

import ru.yandex.client.producer.QueueHostInfo;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.EmptyAsyncConsumerFactory;
import ru.yandex.http.util.nio.HeaderAsyncRequestProducerSupplier;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.http.util.nio.client.AsyncGetURIRequestProducerSupplier;
import ru.yandex.iex.proxy.CacheSentSolutions;
import ru.yandex.iex.proxy.FactsContext;
import ru.yandex.iex.proxy.IexProxy;
import ru.yandex.iex.proxy.Solution;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.async.consumer.JsonAsyncDomConsumerFactory;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.json.xpath.JsonUnexpectedTokenException;
import ru.yandex.json.xpath.ValueUtils;
import ru.yandex.parser.searchmap.User;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.search.prefix.LongPrefix;

public class XivaFactsUpdatingCallback
    extends AbstractFilterFutureCallback<CacheSentSolutions, List<Solution>>
{
    private static final int PRODUCER_RETRY_INTERVAL = 100;

    private static final String UID = "uid";
    private static final String SUID = "suid";
    private static final String MDB = "mdb";
    private static final String MIDS = "mids";
    private static final String XIVA_EVENT_NAME = "iex_widgets_update";
    private static final String LOG_PREFIX = "XivaFactsUpdate ";
    private static final String XIVA_PUSH_REJECTED = ", xiva push rejected";
    private static final String FALSE = "false";
    private static final String TAKSA_HOST = "taksa-host";

    private final FactsContext context;
    private final User user;

    public XivaFactsUpdatingCallback(
        final FactsContext context,
        final FutureCallback<? super List<Solution>> callback)
    {
        super(callback);

        this.context = context;
        String serviceName = context.iexProxy().config().
            factsIndexingQueueName();
        this.user = new User(serviceName, new LongPrefix(context.prefix()));
    }

    @Override
    @SuppressWarnings("FutureReturnValueIgnored")
    public void completed(final CacheSentSolutions result) {
        ProxySession session = context.session();
        if (context.mids() == null || context.mids().isEmpty()) {
            session.logger().info("mids are empty" + XIVA_PUSH_REJECTED);
            callback.completed(result.solutions());
            return;
        }
        // send filter search request
        try {
            final AsyncClient client = context.filterSearchClient();
            final String tvmTicket = context.filterSearchTvm2Ticket();
            final String uri = context.filterSearchUri(context.mids());

            context.session().logger().info(
                LOG_PREFIX + "filter search request: " + uri);
            client.execute(
                new HeaderAsyncRequestProducerSupplier(
                    new AsyncGetURIRequestProducerSupplier(uri),
                    new BasicHeader(
                        YandexHeaders.X_YA_SERVICE_TICKET,
                        tvmTicket)),
                JsonAsyncDomConsumerFactory.OK,
                context.session().listener().createContextGeneratorFor(client),
                new FilterSearchCallback(context, result));
        } catch (NumberFormatException e) {
            context.session().logger().log(
                Level.SEVERE,
                "XivaFactsUpdatingCallback: NumberFormatException for uid: " + context.prefix(),
                e);
            callback.completed(result.solutions());
        } catch (BadRequestException | URISyntaxException e) {
            context.session().logger().log(
                Level.SEVERE,
                "XivaFactsUpdatingCallback: filter search uri syntax error",
                e
            );
            callback.completed(result.solutions());
        }
    }

    @Override
    public void failed(final Exception e) {
        context.session().logger().log(Level.SEVERE, "XivaFactsUpdatingCallback failed", e);
        callback.failed(e);
    }

    private void producerRequest(
        final FactsContext context,
        final ProducerCallback callback)
    {
        context.iexProxy().producerClient().executeWithInfo(
            user,
            context.session().listener().createContextGeneratorFor(
                context.iexProxy().producerClient()),
            callback);
    }

    private final class FilterSearchCallback
        implements FutureCallback<Object>
    {
        private FactsContext context;
        private CacheSentSolutions cachedSolutions;
        private ProxySession session;

        private FilterSearchCallback(
            final FactsContext context,
            final CacheSentSolutions cachedSolutions)
        {
            this.context = context;
            this.cachedSolutions = cachedSolutions;
            this.session = context.session();
        }

        @Override
        public void completed(final Object result) {
            if (result instanceof Map) {
                Object envelopes = ((Map<?, ?>) result).get("envelopes");
                if (envelopes instanceof List
                    && !((List<?>) envelopes).isEmpty())
                {
                    // send taksa request and get widgets info for these mids
                    ProducerCallback callback =
                        new ProducerCallback(context, cachedSolutions, result);
                    producerRequest(context, callback);
                    return;
                }
            }
            session.logger().log(
                Level.WARNING,
                "Filter search result envelopes are empty: " + result);
            callback.completed(cachedSolutions.solutions());
        }

        @Override
        public void failed(final Exception e) {
            logError(session, e);
            callback.completed(cachedSolutions.solutions());
        }

        @Override
        public void cancelled() {
            callback.completed(cachedSolutions.solutions());
        }
    }

    private final class ProducerCallback
        implements FutureCallback<List<QueueHostInfo>>
    {
        private final FactsContext context;
        private final CacheSentSolutions cachedSolutions;
        private final Object envelopes;
        private final long deadline;

        private ProducerCallback(
            final FactsContext context,
            final CacheSentSolutions cachedSolutions,
            final Object envelopes)
        {
            this.context = context;
            this.envelopes = envelopes;
            this.cachedSolutions = cachedSolutions;
            this.deadline =
                System.currentTimeMillis()
                    + context.iexProxy().config()
                    .factsExtractIndexationWaitTimeout();
        }

        @Override
        public void completed(final List<QueueHostInfo> queueHostInfos) {
            Long maxPos = Long.MIN_VALUE;
            for (QueueHostInfo info: queueHostInfos) {
                if (info.queueId() > maxPos) {
                    maxPos = info.queueId();
                }
            }

            context.session().logger().info(
                "Top host position is " + maxPos
                    + " expecting " + cachedSolutions.queueId());
            if (maxPos >= cachedSolutions.queueId()) {
                sendTaksaRequest(envelopes);
            } else {
                if (deadline > System.currentTimeMillis()) {
                    context.iexProxy().producerClient().scheduleRetry(
                        new WaitingBackendPositionTask(this),
                        PRODUCER_RETRY_INTERVAL);
                } else {
                    context.session().logger().warning(
                        "Cannot waite more while extracted facts indexed");
                    callback.completed(cachedSolutions.solutions());
                }
            }
        }

        private void sendTaksaRequest(final Object envelopes) {
            ProxySession session = context.session();
            try {
                QueryConstructor query = new QueryConstructor("/api/list?");
                query.append(UID, context.prefix());
                appendIfNotNull(session, query, SUID);
                appendIfNotNull(session, query, MDB);
                query.append("version", "hound");
                query.append("retry", FALSE);
                String uri = query.toString();
                String body = JsonType.NORMAL.toString(envelopes);

                String taksaHost = session.params().getOrNull(TAKSA_HOST);
                HttpHost taksaTestingHost =
                    context.iexProxy().config().taksaTestingConfig().host();
                HttpHost host;
                AsyncClient client;
                String tvmTicket = null;
                if (taksaTestingHost.getHostName().equals(taksaHost)) {
                    host = taksaTestingHost;
                    client = context.iexProxy().taksaTestingClient();
                } else {
                    host = context.iexProxy().config().taksaConfig().host();
                    client = context.iexProxy().taksaClient();
                    tvmTicket = context.iexProxy().taksaTvm2Ticket();
                }
                client = client.adjust(session.context());
                BasicAsyncRequestProducerGenerator producerGenerator =
                    new BasicAsyncRequestProducerGenerator(uri, body);
                if (tvmTicket != null) {
                    producerGenerator.addHeader(
                        YandexHeaders.X_YA_SERVICE_TICKET,
                        tvmTicket);
                }
                String requestIdHeader =
                    session.headers().getOrNull(YandexHeaders.X_REQUEST_ID);
                if (requestIdHeader != null) {
                    producerGenerator
                        .addHeader(YandexHeaders.X_REQUEST_ID, requestIdHeader);
                }
                String expBoxesHeader =
                    session.headers().getOrNull(YandexHeaders.X_ENABLED_BOXES);
                if (expBoxesHeader != null) {
                    producerGenerator.addHeader(
                        YandexHeaders.X_ENABLED_BOXES,
                        expBoxesHeader);
                }
                session.logger().info(
                    LOG_PREFIX + "taksa request uri: " + host + uri);
                client.execute(
                    host,
                    producerGenerator,
                    JsonAsyncDomConsumerFactory.OK,
                    session.listener().createContextGeneratorFor(client),
                    new TaksaCallback(context, cachedSolutions));
            } catch (BadRequestException e) {
                session.logger().log(Level.SEVERE, "Taksa request error", e);
                callback.completed(cachedSolutions.solutions());
            }
        }

        @Override
        public void failed(final Exception e) {
            context.session().logger().log(
                Level.WARNING,
                "Producer request failed",
                e);

            completed(Collections.emptyList());
        }

        @Override
        public void cancelled() {
            completed(Collections.emptyList());
        }
    }

    private final class WaitingBackendPositionTask extends TimerTask {
        private final ProducerCallback callback;

        private WaitingBackendPositionTask(final ProducerCallback callback) {
            this.callback = callback;
        }

        @Override
        public void run() {
            producerRequest(context, callback);
        }
    }

    private final class TaksaCallback
        implements FutureCallback<Object>
    {
        private final FactsContext context;
        private final CacheSentSolutions cachedSolutions;
        private final ProxySession session;
        private final IexProxy iexProxy;

        TaksaCallback(
            final FactsContext context,
            final CacheSentSolutions cachedSolutions)
        {
            this.context = context;
            this.cachedSolutions = cachedSolutions;
            this.session = context.session();
            this.iexProxy = context.iexProxy();
        }

        @Override
        public void completed(final Object taksaResponse) {
            if (isTaksaResponseEmpty(taksaResponse)) {
                iexProxy.taksaRequestCompleted(true);
                callback.completed(cachedSolutions.solutions());
                return;
            }

            session.logger().info(LOG_PREFIX
                + "taksa response is not empty, send xiva notify");
            iexProxy.taksaRequestCompleted(false);

            ImmutableXivaConfig xivaConfig;
            AsyncClient client;
            if (context.corp()) {
                client = iexProxy.xivaCorpClient();
                xivaConfig = iexProxy.config().xivaCorpConfig();
            } else {
                client = iexProxy.xivaClient();
                xivaConfig = iexProxy.config().xivaConfig();
            }

            try {
                QueryConstructor query = new QueryConstructor("/v2/send?");
                query.append("ttl", 0);
                query.append("token", xivaConfig.token());
                query.append("user", context.prefix());
                query.append("event", XIVA_EVENT_NAME);

                StringBuilderWriter content = new StringBuilderWriter();
                try (JsonWriter writer = new JsonWriter(content)) {
                    writer.startObject();
                    writer.key("payload");
                    writer.value(taksaResponse);
                    writer.endObject();
                }
                session.logger().info(
                    LOG_PREFIX + "xiva request: " + query.toString());
                client.execute(
                    xivaConfig.host(),
                    new BasicAsyncRequestProducerGenerator(
                        query.toString(),
                        content.toString(),
                        ContentType.APPLICATION_JSON),
                    EmptyAsyncConsumerFactory.OK,
                    session.listener().createContextGeneratorFor(client),
                    new XivaCallback(session, cachedSolutions.solutions()));
            } catch (BadRequestException | IOException e) {
                session.logger().log(Level.SEVERE, "Xiva request error", e);
                callback.completed(cachedSolutions.solutions());
            }
        }

        @Override
        public void failed(final Exception e) {
            logError(session, e);
            callback.completed(cachedSolutions.solutions());
        }

        @Override
        public void cancelled() {
            callback.completed(cachedSolutions.solutions());
        }

        private boolean isTaksaResponseEmpty(final Object taksaResponse) {
            Boolean empty = false;
            try {
                Map<?, ?> responseMap = ValueUtils.asMapOrNull(taksaResponse);
                if (taksaResponse != null) {
                    List<?> widgets = ValueUtils.asListOrNull(
                        responseMap.get("widgets"));
                    if (widgets == null || widgets.isEmpty()) {
                        session.logger().info(LOG_PREFIX + "taksa widgets is "
                            + "empty: " + responseMap + XIVA_PUSH_REJECTED);
                        empty = true;
                    }
                } else {
                    session.logger().info(LOG_PREFIX
                        + "taksa response is null" + XIVA_PUSH_REJECTED);
                    empty = true;
                }
            } catch (JsonUnexpectedTokenException e) {
                session.logger().log(
                    Level.SEVERE,
                    LOG_PREFIX + "taksa response parsing error"
                        + XIVA_PUSH_REJECTED,
                    e);
                empty = true;
            }
            return empty;
        }
    }

    private final class XivaCallback implements FutureCallback<Object> {
        private ProxySession session;
        private List<Solution> solutions;

        XivaCallback(
            final ProxySession session,
            final List<Solution> solutions)
        {
            this.session = session;
            this.solutions = solutions;
        }

        @Override
        public void completed(final Object o) {
            session.logger().info("Xiva request completed");
            callback.completed(solutions);
        }

        @Override
        public void failed(final Exception e) {
            logError(session, e);
            callback.completed(solutions);
        }

        @Override
        public void cancelled() {
            callback.completed(solutions);
        }
    }

    private static void logError(
        final ProxySession session,
        final Exception e)
    {
        String details = session.listener().details();
        session.logger().log(
            Level.WARNING,
            "Request failed: " + details + " because of",
            e);
    }

    private static void appendIfNotNull(
        final ProxySession session,
        final QueryConstructor query,
        final String name)
        throws BadRequestException
    {
        String value = session.params().getOrNull(name);
        if (value != null) {
            query.append(name, value);
        }
    }
}
