package ru.yandex.iex.proxy.xutils.mailsender;

import java.util.ArrayList;
import java.util.List;

import ru.yandex.blackbox.BlackboxUserinfo;
import ru.yandex.iex.proxy.AbstractEntityContext;
import ru.yandex.iex.proxy.ImmutableIexForwarderConfig;
import ru.yandex.iex.proxy.xutils.mailsender.limiters.IexForwardLimiter;
import ru.yandex.iex.proxy.xutils.mailsender.limiters.IexForwardLimiterFactory;
import ru.yandex.parser.email.types.MessageType;

public class BasicIexForwarder implements IexForwarder {
    private final ImmutableIexForwarderConfig config;
    private final MailSender mailSender;
    private final List<IexForwardLimiterFactory> factories;
    private final List<IexForwardLimiter> limiters;

    public BasicIexForwarder(
        final MailSender mailSender,
        final ImmutableIexForwarderConfig config)
    {
        this.mailSender = mailSender;
        this.config = config;
        this.factories = config.limiters();
        this.limiters = new ArrayList<>();

        this.reset();
    }

    /**
     * Not Thread Safe! Should be guarded by outer lock
     */
    @Override
    public void reset() {
        this.limiters.clear();

        for (IexForwardLimiterFactory factory: factories) {
            this.limiters.add(factory.create());
        }
    }

    @Override
    public void forward(
        final AbstractEntityContext context,
        final long millsFromDayStart)
    {
        String reason = null;
        if (BlackboxUserinfo.corp(context.prefix())) {
            reason = "No mail forwarding: corp user";
        } else if (context.getTypes().contains(MessageType.PEOPLE)) {
            reason = "No mail forwarding: people mail type";
        } else {
            for (IexForwardLimiter limiter : limiters) {
                if (!limiter.acquire(context, millsFromDayStart)) {
                    StringBuilder sb = new StringBuilder("LimiterFailed ");
                    sb.append(limiter);
                    sb.append(" forwardingTo:");
                    sb.append(config.email());
                    sb.append(" mailFrom:");
                    sb.append(context.getFrom());
                    sb.append(" types:");
                    sb.append(context.getTypes());
                    reason = sb.toString();
                    break;
                }
            }
        }

        if (reason != null) {
            if (config.verboseLog()) {
                context.session().logger().info(reason);
            }
            return;
        }

        if (config.verboseLog()) {
            context.session().logger().info(
                "ForwardingMail " + config.email());
        }

        mailSender.send(new SendmailContext(context, config.email()));
    }
}
