package ru.yandex.iex.proxy.xutils.mailsender;

import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.ZoneOffset;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Consumer;
import java.util.stream.Collectors;

import ru.yandex.iex.proxy.AbstractEntityContext;
import ru.yandex.iex.proxy.ImmutableIexForwarderConfig;

public class MultiIexForwarder implements Consumer<AbstractEntityContext> {
    private static final long ONE_DAY = TimeUnit.DAYS.toMillis(1);

    private final AtomicLong currentDayTs;

    private final List<IexForwarder> forwarders;

    public MultiIexForwarder(
        final MailSender mailSender,
        final List<ImmutableIexForwarderConfig> configs)
    {
        currentDayTs = new AtomicLong(0);
        this.forwarders =
            Collections.unmodifiableList(
                configs.stream()
                    .map((c) -> new BasicIexForwarder(mailSender, c))
                    .collect(Collectors.toList()));
    }

    private synchronized void resetLimiters() {
        for (IexForwarder forwarder: forwarders) {
            forwarder.reset();
        }
    }

    @Override
    public void accept(final AbstractEntityContext context) {
        long requestTime = context.session().connection().requestStartTime();
        long current = currentDayTs.get();
        long msFromDayStart = requestTime - current;
        if (msFromDayStart > ONE_DAY) {
            //ok looks like day changed
            if (currentDayTs.compareAndSet(current, startOfDay(requestTime))) {
                resetLimiters();
            }

            msFromDayStart = requestTime - currentDayTs.get();
        }

        for (IexForwarder forwarder: forwarders) {
            forwarder.forward(context, msFromDayStart);
        }
    }

    public static long startOfDay(final long ts) {
        Date date = new Date(ts);
        LocalDateTime localDateTime =
            LocalDateTime.ofInstant(date.toInstant(), ZoneOffset.UTC);
        LocalDateTime startOfDay = localDateTime.with(LocalTime.MIN);

        return TimeUnit.SECONDS.toMillis(
            startOfDay.atZone(ZoneOffset.UTC).toEpochSecond());
    }
}
