package ru.yandex.iex.proxy.xutils.replace;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.nio.charset.StandardCharsets;
import java.util.HashSet;
import java.util.Locale;
import java.util.Set;

import ru.yandex.iex.proxy.xutils.transliter.SimpleTranslit;

public final class NameChecker {
    private static NameChecker instance;

    private Set<String> names = new HashSet<>();
    private Set<String> surnameSuffixes = new HashSet<>();

    static {
        try {
            instance = new NameChecker();
        } catch (IOException e) {
            throw new RuntimeException("NameChecker creation failed", e);
        }
    }

    private NameChecker() throws IOException {
        SimpleTranslit translit = new SimpleTranslit();
        try (BufferedReader br = new BufferedReader(
            new InputStreamReader(
                NameChecker.class.getResourceAsStream("names.txt"),
                StandardCharsets.UTF_8)))
        {
            String line;
            while ((line = br.readLine()) != null) {
                String rname = line.toLowerCase(Locale.getDefault());
                names.add(rname);
                names.addAll(translit.ruToEnAllOption(rname));
            }
        }
        try (BufferedReader br = new BufferedReader(
            new InputStreamReader(
                NameChecker.class.getResourceAsStream("surname_suffixes.txt"),
                StandardCharsets.UTF_8)))
        {
            String line;
            while ((line = br.readLine()) != null) {
                String rsurname = line.toLowerCase(Locale.getDefault());
                surnameSuffixes.add(rsurname);
                surnameSuffixes.add(translit.ruToEn(rsurname));
            }
        }
    }

    public static NameChecker getInstance() {
        return instance;
    }

    public boolean probableName(final String name) {
        return names.contains(name.toLowerCase(Locale.getDefault()));
    }

    public boolean probableSurname(final String name) {
        if (isNameCase(name)) {
            String lowerName = name.toLowerCase(Locale.getDefault());
            for (String x : surnameSuffixes) {
                if (lowerName.endsWith(x)) {
                    return true;
                }
            }
        }
        return false;
    }

    public boolean probablePatronymic(final String pat) {
        return isNameCase(pat) && pat.toLowerCase(
            Locale.getDefault()).matches(".*(TCH|ICH|ich|tch|вич|вна|vna)");
    }

    private boolean isNameCase(final String name) {
        if (name == null || name.isEmpty()) {
            return false;
        }
        int up = 0;
        for (int i = 1; i < name.length(); ++i) {
            if (Character.isUpperCase(name.charAt(i))) {
                ++up;
            }
        }
        return Character.isUpperCase(name.charAt(0))
            && (up == 0 || up == name.length() - 1);
    }
}
