package ru.yandex.iex.proxy.xutils.spamreport;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.text.SimpleDateFormat;
import java.util.Arrays;
import java.util.Date;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.logging.Level;
import java.util.stream.Collectors;

import org.apache.http.HttpEntity;
import org.apache.http.HttpStatus;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.mime.FormBodyPartBuilder;
import org.apache.http.entity.mime.MultipartEntityBuilder;
import org.apache.http.entity.mime.content.ByteArrayBody;
import org.apache.http.entity.mime.content.ContentBody;
import org.apache.http.entity.mime.content.StringBody;

import ru.yandex.function.ByteArrayProcessable;
import ru.yandex.function.ByteArrayProcessor;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.ByteArrayProcessableAsyncConsumerFactory;
import ru.yandex.http.util.nio.EmptyAsyncConsumerFactory;
import ru.yandex.http.util.nio.StatusCheckAsyncResponseConsumerFactory;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.iex.proxy.IexProxy;
import ru.yandex.io.ByteArrayInputStreamFactory;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.writer.JsonType;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.util.storage.DataExtractorConfigBuilder;
import ru.yandex.util.storage.ImmutableDataExtractorConfig;
import ru.yandex.util.storage.MailTextExtractor;
import ru.yandex.util.storage.StorageData;
import ru.yandex.util.string.StringUtils;

public class SpamReportSender {
    private static final ImmutableDataExtractorConfig DATA_EXTRACTOR_CONFIG =
        new DataExtractorConfigBuilder()
            .maxInputLength(65536)
            .truncateLongInput(true)
            .charset(null)
            .build();
    private static final MailTextExtractor FAST_TEXT_EXTRACTOR =
        new MailTextExtractor() {
            @Override
            protected boolean findFirstHtml() {
                return true;
            }
        };

    public static void send(
        final SendMailContext context,
        final FutureCallback<Void> callback,
        final boolean spam,
        final List<JsonObject> neuroHardResult)
    {
        sendGetEmlRequest(
            context.iexProxy(),
            context.session(),
            context.stid(),
            new EmlCallback(context, callback, spam, neuroHardResult));
    }

    public static void sendGetEmlRequest(
        final IexProxy iexProxy,
        final ProxySession session,
        final String stid,
        final FutureCallback<ByteArrayProcessable> callback)
    {
        final AsyncClient client =
            iexProxy.mulcagateClient().adjust(session.context());

        BasicAsyncRequestProducerGenerator producerGenerator =
            new BasicAsyncRequestProducerGenerator(
                "/gate/get/" + stid);
        producerGenerator.addHeader(
            YandexHeaders.X_YA_SERVICE_TICKET,
            iexProxy.unistorageTvm2Ticket());

        client.execute(
            iexProxy.mulcagateHost(),
            producerGenerator,
            new StatusCheckAsyncResponseConsumerFactory<ByteArrayProcessable>(
                x -> x < HttpStatus.SC_BAD_REQUEST
                || x == HttpStatus.SC_NOT_FOUND
                || x == HttpStatus.SC_LOCKED,
                ByteArrayProcessableAsyncConsumerFactory.INSTANCE),
            session.listener().createContextGeneratorFor(client),
            callback);
    }

    private static class EmlCallback
        extends AbstractFilterFutureCallback<ByteArrayProcessable, Void>
    {
        private final SendMailContext context;
        private final boolean spam;
        private final List<JsonObject> neuroHardResult;

        EmlCallback(
            final SendMailContext context,
            final FutureCallback<Void> callback,
            final boolean spam,
            final List<JsonObject> neuroHardResult)
        {
            super(callback);
            this.context = context;
            this.spam = spam;
            this.neuroHardResult = neuroHardResult;
        }

        @Override
        public void completed(final ByteArrayProcessable eml) {
            try {
                QueryConstructor qc = new QueryConstructor("/send-mail?");
                String rfcDate =
                    new SimpleDateFormat(
                        "E, d MMM yyyy HH:mm:ss Z",
                        Locale.US)
                            .format(new Date(context.date() * 1000));
                String humanDate =
                    new SimpleDateFormat(
                        "yyyy.MM.dd HH:mm:ss",
                        Locale.US)
                            .format(new Date(context.date() * 1000));

                qc.append("to", "so-compains@yandex.ru");
                qc.append(
                    "msg-to",
                    context.to() + ", "
                        + context.prefix() + "@uid.ya");
                qc.append("from", context.from());
                String subj = humanDate;
                if (spam) {
                    subj += " [Spam]";
                }
                subj += ": " + context.subject();
                qc.append("subject", subj);
                qc.append(
                    "header",
                    "Date: " + rfcDate);
                qc.append(
                    "header",
                    "X-Yandex-Timestamp: " + context.date());
                qc.append(
                    "header",
                    "Message-Id: " + context.msgId());
                qc.append(
                    "header",
                    "x-yandex-complainer-uid: " + context.prefix());
                qc.append(
                    "header",
                    "x-yandex-stid: " + context.stid());
                qc.append(
                    "header",
                    "x-yandex-complain-mid: " + context.mid());
                if (neuroHardResult != null && !neuroHardResult.isEmpty()) {
                    qc.append(
                        "header",
                        "x-neuro-hard: "
                        + JsonType.NORMAL.toString(neuroHardResult));
                }
                if (context.additionalHeaders() != null) {
                    for (Map.Entry<String, String> i : context.additionalHeaders()) {
                        qc.append(
                            "header",
                            i.getKey() + ": " + i.getValue());
                    }
                }
                qc.append(
                    "types",
                    StringUtils.join(
                        context.types().stream().map(Object::toString)
                            .collect(Collectors.toList()),
                        ','));

                ContentBody text = new StringBody("No data extracted");
                try {
                    StorageData data = FAST_TEXT_EXTRACTOR.extractDataFromEml(
                        eml.processWith(ByteArrayInputStreamFactory.INSTANCE),
                        DATA_EXTRACTOR_CONFIG);
                    if (data == null) {
                        text = new StringBody("No text found in " + context.stid());
                    } else {
                        text =
                            data.processWith(
                                new ConfigurableByteArrayBodyFactory(
                                    data.contentType(),
                                    "text"));
                    }
                } catch (Exception e) {
                    StringBuilderWriter sbw = new StringBuilderWriter();
                    e.printStackTrace(sbw);
                    text = new StringBody(sbw.toString());
                }

                MultipartEntityBuilder builder =
                    MultipartEntityBuilder.create();
                builder.setStrictMode();
                builder.setCharset(StandardCharsets.UTF_8);
                builder.setMimeSubtype("mixed");
                builder.setBoundary("my_lucky_boundary");
                builder.addPart(
                    FormBodyPartBuilder.create()
                        .setBody(text)
                        .addField("Content-Disposition", "inline")
                        .setName("text")
                        .build());
                builder.addPart(
                    FormBodyPartBuilder.create()
                        .setBody(
                            eml.processWith(ByteArrayBodyFactory.INSTANCE))
                        .setName("mail.eml")
                        .addField(
                            "Content-Disposition",
                            "attachment; filename=mail.eml")
                        .build());
                HttpEntity entity = builder.build();
                qc.append(
                    "content-type",
                    entity.getContentType().getValue());
                qc.append("raw", "true");

                BasicAsyncRequestProducerGenerator generator =
                    new BasicAsyncRequestProducerGenerator(
                        qc.toString(),
                        builder.build());
                AsyncClient client =
                    context.iexProxy().gatemailClient().adjust(
                        context.session().context());
                client.execute(
                    context.iexProxy().config()
                        .gatemailConfig().host(),
                    generator,
                    EmptyAsyncConsumerFactory.NON_FATAL,
                    context.session().listener()
                        .createContextGeneratorFor(client),
                    callback);
            } catch (BadRequestException|IOException e) {
                context.session().logger().log(
                    Level.WARNING,
                    "SpamReportSender: Failed to construct sendmail context",
                    e);
            }
        }
    }

    public enum ByteArrayBodyFactory
        implements ByteArrayProcessor<ByteArrayBody, RuntimeException>
    {
        INSTANCE;

        @Override
        public ByteArrayBody process(
            final byte[] buf,
            final int off,
            final int len)
        {
            return new ByteArrayBody(
                Arrays.copyOfRange(buf, off, off + len),
                ContentType.create("message/rfc822"),
                "mail.eml");
        }
    }

    private static class ConfigurableByteArrayBodyFactory
        implements ByteArrayProcessor<ByteArrayBody, RuntimeException>
    {
        private final ContentType contentType;
        private final String name;

        ConfigurableByteArrayBodyFactory(
            final ContentType contentType,
            final String name)
        {
            this.contentType = contentType;
            this.name = name;
        }

        @Override
        public ByteArrayBody process(
            final byte[] buf,
            final int off,
            final int len)
        {
            return new ByteArrayBody(
                Arrays.copyOfRange(buf, off, off + len),
                contentType,
                name);
        }
    }
}
