package matchers

import (
	"fmt"
	"reflect"
	"strings"
	"testing"
)

type Matcher interface {
	Match(interface{}) bool
	String() string
}

type Mortal interface {
	Fatal(args ...interface{})
}

type Protect struct {
	*testing.T
}

func (t Protect) Fatal(args ...interface{}) {
	t.Error(args...)
}

type EqualTo struct {
	V interface{}
}

func (m EqualTo) Match(i interface{}) bool {
	return reflect.DeepEqual(m.V, i)
}

func (m EqualTo) String() string {
	return fmt.Sprintf("equal to %v(%v)", reflect.TypeOf(m.V), m.V)
}

type Is struct {
	V interface{}
}

func (m Is) matcher() Matcher {
	switch m.V.(type) {
	case Matcher:
		return m.V.(Matcher)
	}
	return &EqualTo{m.V}
}

func (m Is) Match(i interface{}) bool {
	return m.matcher().Match(i)
}

func (m Is) String() string {
	return fmt.Sprintf("is %v", m.matcher())
}

type TypeOf struct {
	V interface{}
}

func (m TypeOf) Match(i interface{}) bool {
	return reflect.TypeOf(m.V) == reflect.TypeOf(i)
}

func (m TypeOf) String() string {
	return fmt.Sprintf("type %v", reflect.TypeOf(m.V))
}

type Not struct {
	V interface{}
}

func (m Not) Match(i interface{}) bool {
	return !Is(m).Match(i)
}

func (m Not) String() string {
	return fmt.Sprintf("not %v", m.V)
}

type AllOf []Matcher

func (all AllOf) Match(v interface{}) bool {
	for _, m := range all {
		if !m.Match(v) {
			return false
		}
	}
	return true
}

func (all AllOf) String() string {
	s := []string{}
	for _, m := range all {
		s = append(s, fmt.Sprintf("%v", m))
	}
	return strings.Join(s, ", and ")
}

type AnyOf []Matcher

func (any AnyOf) Match(v interface{}) bool {
	for _, m := range any {
		if m.Match(v) {
			return true
		}
	}
	return false
}

func (any AnyOf) String() string {
	s := ""
	for i, m := range any {
		s += fmt.Sprintf("%v", m)
		if i < len(any)-1 {
			s += ", or "
		}
	}
	return s
}

type ElementsAre []interface{}

func (elementsAre ElementsAre) Match(vs interface{}) bool {
	if reflect.TypeOf(vs).Kind() == reflect.Slice {
		return elementsAre.match(reflect.ValueOf(vs))
	}
	return false
}

func (elementsAre ElementsAre) matcher(i int) Matcher {
	switch elementsAre[i].(type) {
	case Matcher:
		return elementsAre[i].(Matcher)
	}
	return &EqualTo{elementsAre[i]}
}

func (elementsAre ElementsAre) match(vs reflect.Value) bool {
	if len(elementsAre) != vs.Len() {
		return false
	}
	m := make(map[int]bool)
	for i := range elementsAre {
		for j := 0; j < vs.Len(); j++ {
			if elementsAre.matcher(i).Match(vs.Index(j).Interface()) {
				m[i] = true
			}
		}
	}
	return len(m) == len(elementsAre)
}

func (elementsAre ElementsAre) String() string {
	s := []string{}
	for _, m := range elementsAre {
		s = append(s, fmt.Sprintf("%v", m))
	}
	return "elements are: [" + strings.Join(s, ", ") + "]"
}

type Contains []interface{}

func (contains Contains) Match(vs interface{}) bool {
	if reflect.TypeOf(vs).Kind() == reflect.Slice {
		return contains.match(reflect.ValueOf(vs))
	}
	return false
}

func (contains Contains) matcher(i int) Matcher {
	switch contains[i].(type) {
	case Matcher:
		return contains[i].(Matcher)
	}
	return &EqualTo{contains[i]}
}

func (contains Contains) match(vs reflect.Value) bool {
	for i := range contains {
		match := false
		for j := 0; j < vs.Len(); j++ {
			if contains.matcher(i).Match(vs.Index(j).Interface()) {
				match = true
			}
		}
		if !match {
			return false
		}
	}
	return true
}

func (contains Contains) String() string {
	s := []string{}
	for _, m := range contains {
		s = append(s, fmt.Sprintf("%v", m))
	}
	return "contains: [" + strings.Join(s, ", ") + "]"
}

type Fails struct {
}

func (m Fails) Match(i interface{}) bool {
	err := i.(Expect).Confirm()
	return err != nil
}

func (m Fails) String() string {
	return "fails"
}

type Expect struct {
	I interface{}
	M Matcher
}

func (e Expect) String() string {
	return fmt.Sprintf("%v %v", e.I, e.M)
}

func (e Expect) Confirm() error {
	if !e.M.Match(e.I) {
		return fmt.Errorf("%v(%v) %v", reflect.TypeOf(e.I), e.I, e.M)
	}
	return nil
}

func AssertThat(t Mortal, i interface{}, m Matcher) {
	err := Expect{i, m}.Confirm()
	if err != nil {
		t.Fatal(fmt.Sprintf("expect that: %v", err))
	}
}
