package ru.yandex.sobb.front;

import org.apache.http.HttpHeaders;
import org.apache.http.HttpHost;
import org.apache.http.HttpStatus;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.collection.LongPair;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.ServerException;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.parser.uri.QueryConstructor;


/**
 * Reads headers X_REAL_IP, HOST and SESSION_ID,
 * sends request to blackbox.
 */
public class Authorizer {
    private final AsyncClient blackbox;
    private final HttpHost blackboxHost;

    private final static String BEARER = "Bearer ";

    public Authorizer(AsyncClient blackbox, HttpHost blackboxHost) {
        this.blackbox = blackbox;
        this.blackboxHost = blackboxHost;
    }

    public void request(
        ProxySession session,
        FutureCallback<LongPair<String>> callback)
        throws BadRequestException
    {
        String realIp = session.headers()
            .getString(YandexHeaders.X_REAL_IP, "127.0.0.1");

        String oauthToken = session.params().getOrNull("oauth_token");
        if (oauthToken == null) {
            String auth = session.headers()
                .getOrNull(HttpHeaders.AUTHORIZATION);
            if (auth != null && auth.startsWith(BEARER)) {
                oauthToken = auth.substring(BEARER.length());
            }
        }
        if (oauthToken == null) {
            callback.failed(
                new ServerException(HttpStatus.SC_UNAUTHORIZED,
                "oauth_token not specified"));
            return;
        }

        QueryConstructor qc = new QueryConstructor(
            "/blackbox?method=oauth"
            + "&get_user_ticket=yes"
            + "&format=json");
        qc.append("userip", realIp);
        qc.append("oauth_token", oauthToken);

        AsyncClient client = blackbox.adjust(session.context());
        client.execute(
            blackboxHost,
            new BasicAsyncRequestProducerGenerator(qc.toString()),
            JsonAsyncTypesafeDomConsumerFactory.OK,
            session.listener().createContextGeneratorFor(client),
            new BlackboxParsingCallback(callback));
    }

    private static class BlackboxParsingCallback
        extends AbstractFilterFutureCallback<JsonObject, LongPair<String>>
    {
        protected BlackboxParsingCallback(
            FutureCallback<LongPair<String>> callback)
        {
            super(callback);
        }

        @Override
        public void completed(JsonObject blackboxResponse) {
            try {
                Integer status = blackboxResponse.asMap()
                    .getMap("status")
                    .getInt("id", 6);
                Long uid = blackboxResponse.asMap()
                    .getMap("uid")
                    .getLong("value", null);
                String userTicket = blackboxResponse.asMap()
                    .getString("user_ticket", null);
                if (status != 0 || uid == null) { // 0 is VALID
                    callback.failed(
                        new ServerException(HttpStatus.SC_UNAUTHORIZED));
                    return;
                }
                callback.completed(new LongPair<>(uid, userTicket));
            } catch (JsonException e) {
                callback.failed(
                    new ServerException(HttpStatus.SC_UNAUTHORIZED));
            }
        }
    }
}
