package ru.yandex.sobb.front;

import java.io.IOException;
import java.util.logging.Level;

import org.apache.http.HttpException;
import org.apache.http.HttpHost;
import org.apache.http.HttpRequest;
import org.apache.http.HttpResponse;
import org.apache.http.HttpStatus;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.nio.protocol.HttpAsyncExchange;
import org.apache.http.nio.protocol.HttpAsyncRequestConsumer;
import org.apache.http.nio.protocol.HttpAsyncRequestHandler;
import org.apache.http.protocol.HttpContext;

import ru.yandex.collection.LongPair;
import ru.yandex.collection.Pattern;
import ru.yandex.http.proxy.BasicProxySession;
import ru.yandex.http.proxy.HttpProxy;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.ServerException;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.nio.AsyncStringConsumer;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.BasicAsyncResponseConsumerFactory;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.http.util.request.RequestHandlerMapper;
import ru.yandex.parser.config.ConfigException;

public class Server<Config> extends HttpProxy {
    private final Authorizer authorizer;

    public Server(final Config config) throws IOException {
        super(config);
        this.authorizer = new Authorizer(
            client("blackbox", config.blackbox()),
            config.blackbox().host());

        register(
            new Pattern<>("/uuids", false),
            new GetUuidsHandler(
                new UuidsProducer(config.secret(), config.uuidsPerUser())),
            RequestHandlerMapper.GET);
        register(
            new Pattern<>("/score", false),
            new GetScoreHandler(
                client("msearch", config.msearch()),
                config.msearch().host()),
            RequestHandlerMapper.GET);
    }

    public static void main(final String... args)
        throws ConfigException, IOException
    {
        main(new ServerFactory(), args);
    }

    private class GetUuidsHandler implements HttpAsyncRequestHandler<String> {
        private final UuidsProducer uuidsProducer;

        private GetUuidsHandler(UuidsProducer uuidsProducer) {
            this.uuidsProducer = uuidsProducer;
        }

        @Override
        public HttpAsyncRequestConsumer<String>
        processRequest(HttpRequest r, HttpContext c) {
            return new AsyncStringConsumer();
        }

        @Override
        public void handle(
            String ignored,
            HttpAsyncExchange exchange,
            HttpContext context)
            throws HttpException
        {
            ProxySession session =
                new BasicProxySession(Server.this, exchange, context);
            authorizer.request(session, new UuidsProducingCallback(session));
        }

        private class UuidsProducingCallback
            extends SafeProxySessionCallback<LongPair<String>>
        {
            public UuidsProducingCallback(ProxySession session) {
                super(session);
            }

            @Override
            public void completed(LongPair<String> uidAndTicket) {
                session.response(
                    HttpStatus.SC_OK,
                    String.join(
                        "\n",
                        uuidsProducer.generate(uidAndTicket.first())));
            }
        }
    }

    private class GetScoreHandler implements HttpAsyncRequestHandler<String> {
        private final AsyncClient msearch;
        private final HttpHost msearchHost;

        private GetScoreHandler(AsyncClient msearch, HttpHost msearchHost) {
            this.msearch = msearch;
            this.msearchHost = msearchHost;
        }

        @Override
        public HttpAsyncRequestConsumer<String>
        processRequest(HttpRequest r, HttpContext c) {
            return new AsyncStringConsumer();
        }

        @Override
        public void handle(
            String ignored,
            HttpAsyncExchange exchange,
            HttpContext context)
            throws HttpException
        {
            ProxySession session =
                new BasicProxySession(Server.this, exchange, context);
            authorizer.request(session,
                new MsearchRequestingCallback(
                    session,
                    new SafeProxySessionCallback<>(session) {
                        @Override
                        public void completed(HttpResponse response) {
                            session.response(response);
                        }
                    }));
        }

        private class MsearchRequestingCallback extends
                AbstractFilterFutureCallback<LongPair<String>, HttpResponse>
        {
            private final ProxySession session;

            protected MsearchRequestingCallback(
                ProxySession session,
                FutureCallback<HttpResponse> callback)
            {
                super(callback);
                this.session = session;
            }

            @Override
            public void completed(LongPair<String> uidAndTicket) {
                AsyncClient client = msearch.adjust(session.context());
                BasicAsyncRequestProducerGenerator request =
                    new BasicAsyncRequestProducerGenerator(
                        "/api/async/so/sobb-get-score");
                request.addHeader(
                    YandexHeaders.X_YA_USER_TICKET,
                    uidAndTicket.second());
                client.execute(
                    msearchHost,
                    request,
                    BasicAsyncResponseConsumerFactory.OK,
                    session.listener().createContextGeneratorFor(client),
                    callback);
            }
        }
    }

    /**
     * Like AbstractProxySessionCallback, but does not print stacktrace on fail
     */
    private abstract static class SafeProxySessionCallback<T>
        implements FutureCallback<T>
    {
        protected final ProxySession session;

        protected SafeProxySessionCallback(ProxySession session) {
            this.session = session;
        }

        @Override
        public void cancelled() {
            session.logger().warning(
                "Request cancelled: " + session.listener().details());
        }

        @Override
        public void failed(final Exception e) {
            String details = session.listener().details();
            session.logger().log(
                Level.WARNING,
                "Request failed: " + details + " because of",
                e);

            // Do not leak stacktrace
            if (e instanceof ServerException) {
                session.response(((ServerException) e).statusCode());
            } else {
                session.response(HttpStatus.SC_INTERNAL_SERVER_ERROR);
            }
        }
    }

}
