package client

import (
	"fmt"
	"io/ioutil"
	"net"
	"net/http"
	"time"

	"a.yandex-team.ru/mail/iex/taksa/logger"
)

var Instance *http.Client = nil
var MaxTimeoutInstance *http.Client = nil

var connectTimeout time.Duration

func dial(network, addr string) (conn net.Conn, err error) {
	start := time.Now()

	conn, err = net.DialTimeout(network, addr, connectTimeout)
	if err != nil {
		logger.Error("", "", "client", "resolve/connect error", extra(addr, err, start))
	}

	return
}

func extra(host string, err error, start time.Time) (extra logger.Extra) {
	extra = logger.Extra{
		"host": host,
		"time": fmt.Sprintf("%.03f", time.Since(start).Seconds()),
	}

	if err != nil {
		extra["err"] = fmt.Sprintf("[%q]", err.Error())
	}

	return
}

func Init(cfg Config) {
	connectTimeout = time.Duration(cfg.ConnectTimeout)
	transport := &http.Transport{Dial: dial, MaxIdleConns: 200, MaxIdleConnsPerHost: 100}
	Instance = &http.Client{Timeout: time.Duration(cfg.Timeout), Transport: transport}
	MaxTimeoutInstance = &http.Client{Timeout: time.Duration(cfg.MaxTimeout), Transport: transport}
}

type Impl struct {
	native *http.Client
	log    logger.Interface
}

func New(l logger.Interface, sync bool) *Impl {
	if sync {
		if MaxTimeoutInstance == nil {
			panic(fmt.Sprintln("[taksa] Fatal: max timeout client not initialized"))
		}
		return &Impl{native: MaxTimeoutInstance, log: l}
	} else {
		if Instance == nil {
			panic(fmt.Sprintln("[taksa] Fatal: client not initialized"))
		}
		return &Impl{native: Instance, log: l}
	}
}

type CodeError struct {
	Code int
}

func (err CodeError) Error() string {
	return fmt.Sprintf("%v", err.Code)
}

func (impl *Impl) get(params Params) (response string, err error) {
	impl.log.InfoExtra("client", "get", logger.Extra{"host": params.URL})
	start := time.Now()

	req, _ := http.NewRequest("GET", params.URL, nil)
	for k, v := range params.Headers {
		req.Header.Add(k, v)
	}

	resp, err := impl.native.Do(req)
	if err != nil {
		impl.log.ErrorExtra("client", "request error", extra(params.URL, err, start))
		return
	}

	defer func() { _ = resp.Body.Close() }()

	if resp.StatusCode != http.StatusOK {
		err = CodeError{resp.StatusCode}
		impl.log.ErrorExtra("client", "non-200 response", extra(params.URL, err, start))
		return
	}

	body, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		impl.log.ErrorExtra("client", "fetch body error", extra(params.URL, err, start))
		return
	}

	response = string(body)
	impl.log.InfoExtra("client", "got", extra(params.URL, err, start))

	return
}
