package currency

import (
	"a.yandex-team.ru/mail/iex/taksa/client"
	"a.yandex-team.ru/mail/iex/taksa/logger"
	"encoding/xml"
	"fmt"
	"time"
)

type Impl struct {
	Cfg Config
	Log logger.Interface
	Cli client.Interface
}

type Error string

func (err Error) Error() string {
	return "currency error: " + string(err)
}

type Currency struct {
	XMLName          xml.Name `xml:"stock"`
	Sdts             []Sdt    `xml:"sdt"`
	NumeratorScale   float64  `xml:"numerator_scale"`
	DenominatorScale float64  `xml:"denominator_scale"`
}

type Sdt struct {
	Value float64 `xml:"value"`
}

func (impl Impl) Get(from, to CountryGeoID) (res string, err error) {
	if from == 0 || to == 0 {
		err = Error("invalid geoid")
		impl.Log.Info("currency", "zero geoid")
		return
	}
	if from == to {
		err = Error("no currency exchange rates for domestic flights")
		return
	}
	toCurrency, err := GetGeoCurrencyInstance().GetCurrencyByGeoid(to)
	if err != nil {
		impl.Log.Error("currency", err.Error())
		return
	}
	fromCurrency, err := GetGeoCurrencyInstance().GetCurrencyByGeoid(from)
	if err != nil {
		impl.Log.Error("currency", err.Error())
		return
	}
	quote := toCurrency + "/" + fromCurrency
	quoteID, err := GetQuoteIdsInstance().GetQuoteID(quote)
	if err != nil {
		impl.Log.Error("currency", err.Error())
		return
	}
	url := fmt.Sprintf("%v/%v.xml", impl.Cfg.Host, quoteID)
	headers := map[string]string{"Host": impl.Cfg.HeaderHost}
	got, err := impl.Cli.Get(client.Params{URL: url, Headers: headers, Timeout: time.Duration(impl.Cfg.Timeout)})
	if err != nil {
		impl.Log.Error("currency", "http error")
		return
	}
	var response Currency
	if err = xml.Unmarshal([]byte(got), &response); err != nil {
		impl.Log.ErrorExtra("currency", "xml parse error", logger.Extra{"err": err.Error()})
		return
	}
	currency, err := impl.GetCurrency(response)
	if err != nil {
		impl.Log.Error("currency", err.Error())
		return
	}
	res = fmt.Sprintf("%v=%.2f %v", toCurrency, currency, fromCurrency)
	return
}

func (impl Impl) GetCurrency(currency Currency) (result float64, err error) {
	if len(currency.Sdts) == 0 {
		err = Error("sdts are empty")
		return
	}
	value := currency.Sdts[0].Value
	if value == 0 {
		err = Error("no value")
		return
	}
	numScale := currency.NumeratorScale
	if numScale == 0 {
		numScale = 1
	}
	denScale := currency.DenominatorScale
	if denScale == 0 {
		denScale = 1
	}
	value = value / numScale * denScale
	if value < 0.01 {
		err = Error("rate is very small")
		return
	}
	return value, nil
}
