package daria

import (
	"encoding/json"
	"sort"
	"strings"
	"time"

	"a.yandex-team.ru/mail/iex/taksa/errs"
	"a.yandex-team.ru/mail/iex/taksa/logger"
	"a.yandex-team.ru/mail/iex/taksa/meta"
	"a.yandex-team.ru/mail/iex/taksa/tutil"
)

type Response struct {
	Messages Messages `json:"messages"`
}

func (response Response) valid() bool {
	if len(response.Messages.List.Message) == 0 {
		return false
	}
	for _, i := range response.Messages.List.Message {
		if !i.valid() {
			return false
		}
	}
	return true
}

type Messages struct {
	List List `json:"list"`
}

type MessageArr []Message

type List struct {
	Message MessageArr `json:"message"`
}

func (messageArr *MessageArr) UnmarshalJSON(buf []byte) error {
	arr := []Message{}
	err := json.Unmarshal(buf, &arr)
	if err == nil {
		*messageArr = append(*messageArr, arr...)
		return nil
	}
	msg := Message{}
	err = json.Unmarshal(buf, &msg)
	if err == nil {
		*messageArr = append(*messageArr, msg)
		return nil
	}
	return err
}

type Message struct {
	Mid       string      `json:"id"`
	Subject   interface{} `json:"subject"`
	Firstline string      `json:"firstline"`
	Types     string      `json:"types"`
	From      Recipient   `json:"from"`
	Date      DateStruct  `json:"date"`
}

type Recipient struct {
	Name  string `json:"name"`
	Email string `json:"email"`
}

type DateStruct struct {
	TS interface{} `json:"utc_timestamp"`
}

func (message Message) valid() bool {
	return message.Mid != ""
}

func getDisplayName(m Message) (res string) {
	res = m.From.Name
	if res == "" {
		res = m.From.Email
	}
	return
}

func getSortedTypes(m Message) (res sort.IntSlice) {
	typeStrings := strings.Split(m.Types, "\u000a")
	typeInts := tutil.Strings2Ints(typeStrings)
	res = sort.IntSlice(typeInts)
	res.Sort()
	return
}

func getSubject(i interface{}) string {
	switch v := i.(type) {
	case string:
		return v
	case map[string]string:
		return v["$t"]
	default:
		return ""
	}
}

func getReceivedDate(i interface{}) (res time.Time) {
	res, err := tutil.GetTimeFromSomething(i)
	if err != nil {
		res = time.Now()
	}
	return
}

func GetEnvelopes(body []byte, l logger.Interface) (res []meta.Envelope, e error) {
	var response Response
	if err := json.Unmarshal(body, &response); err != nil {
		l.ErrorExtra("list", "cannot parse body json", logger.Extra{"err": err.Error()})
		e = errs.BadRequest{Err: "bad json in body"}
		return
	}
	if !response.valid() {
		l.Error("list", "body json lacks mid")
		e = errs.BadRequest{Err: "invalid body"}
		return
	}
	for _, i := range response.Messages.List.Message {
		res = append(res, meta.Envelope{
			Mid:             i.Mid,
			Subject:         getSubject(i.Subject),
			Firstline:       i.Firstline,
			Types:           getSortedTypes(i),
			FromDisplayName: getDisplayName(i),
			FromAddress:     i.From.Email,
			ReceivedDate:    getReceivedDate(i.Date.TS),
			ToAddresses:     []string{}})
	}
	return
}
