package future

import (
	"time"
)

type Promise interface {
	Set(v interface{}, e error)
}

type Future interface {
	Get() (interface{}, error)
	GetTimed(time.Duration) (interface{}, error)
}

func NewPromiseFuture() (Promise, Future) {
	ch := make(channel, 1)
	return &promise{&ch}, &future{&ch}
}

type result struct {
	v interface{}
	e error
}

func (res result) Get() (interface{}, error) {
	return res.v, res.e
}

type channel chan result

type promise struct {
	ch *channel
}

func (pr *promise) Set(v interface{}, e error) {
	if pr.ch != nil {
		*(pr.ch) <- result{v, e}
		pr.ch = nil
	}
}

type badFuture struct {
	err Error
}

func (bf *badFuture) Get() (interface{}, error) {
	return nil, bf.err
}

func (bf *badFuture) GetTimed(_ time.Duration) (interface{}, error) {
	return bf.Get()
}

type future struct {
	ch *channel
}

func (f *future) reset() {
	f.ch = nil
}

func (f *future) get(selector func() (interface{}, error)) (interface{}, error) {
	if f.ch == nil {
		return nil, Error("Promise is not set")
	}
	defer f.reset()
	return selector()
}

func (f *future) Get() (interface{}, error) {
	return f.get(func() (interface{}, error) {
		return (<-*(f.ch)).Get()
	})
}

func (f *future) GetTimed(timeout time.Duration) (interface{}, error) {
	return f.get(func() (interface{}, error) {
		select {
		case res := <-*(f.ch):
			return res.Get()
		case <-time.After(timeout):
		}
		return nil, TimeoutError(timeout.String())
	})
}
