package hound

import (
	"encoding/json"
	"html"
	"sort"
	"time"

	"a.yandex-team.ru/mail/iex/taksa/errs"
	"a.yandex-team.ru/mail/iex/taksa/logger"
	"a.yandex-team.ru/mail/iex/taksa/meta"
	"a.yandex-team.ru/mail/iex/taksa/tutil"
)

type Threads struct {
	List List `json:"threads_by_folder"`
}

type List struct {
	Envelopes []Envelope `json:"envelopes"`
}

func (list List) valid() bool {
	if len(list.Envelopes) == 0 {
		return false
	}
	for _, i := range list.Envelopes {
		if !i.valid() {
			return false
		}
	}
	return true
}

type Envelope struct {
	Mid       string      `json:"mid"`
	Subject   string      `json:"subject"`
	Firstline string      `json:"firstline"`
	Types     []int       `json:"types"`
	From      []Recipient `json:"from"`
	To        []Recipient `json:"to"`
	Date      interface{} `json:"date"`
}

type Recipient struct {
	DisplayName string `json:"displayName"`
	Local       string `json:"local"`
	Domain      string `json:"domain"`
}

func (envelope Envelope) valid() bool {
	return envelope.Mid != ""
}

func getDisplayName(env Envelope) (res string) {
	if len(env.From) > 0 {
		res = env.From[0].DisplayName
		if res == "" {
			res = env.From[0].Local + "@" + env.From[0].Domain
		}
	}
	return
}

func getFromAddress(env Envelope) (res string) {
	if len(env.From) > 0 {
		if len(env.From[0].Local) > 0 && len(env.From[0].Domain) > 0 {
			res = env.From[0].Local + "@" + env.From[0].Domain
		}
	}
	return
}

func getToAddresses(env Envelope) (res []string) {
	if len(env.To) < 1 {
		return []string{}
	}
	for _, rcpt := range env.To {
		res = append(res, rcpt.Local+"@"+rcpt.Domain)
	}
	return
}

func getReceivedDate(i interface{}) (res time.Time) {
	res, err := tutil.GetTimeFromSomething(i)
	if err != nil {
		res = time.Now()
	}
	return
}

func GetEnvelopes(body []byte, l logger.Interface) (res []meta.Envelope, e error) {
	var list List
	if err := json.Unmarshal(body, &list); err != nil {
		l.ErrorExtra("list", "cannot parse body json", logger.Extra{"err": err.Error()})
		e = errs.BadRequest{Err: "bad json in body"}
		return
	}
	if !list.valid() {
		var threads Threads
		if err := json.Unmarshal(body, &threads); err != nil {
			l.ErrorExtra("list", "cannot parse body json", logger.Extra{"err": err.Error()})
			e = errs.BadRequest{Err: "bad json in body"}
			return
		}
		list = threads.List
	}
	if !list.valid() {
		l.Error("list", "body json lacks mid")
		e = errs.BadRequest{Err: "invalid body"}
		return
	}
	for _, i := range list.Envelopes {
		sorted := sort.IntSlice(i.Types)
		sorted.Sort()
		res = append(res, meta.Envelope{
			Mid:             i.Mid,
			Subject:         i.Subject,
			Firstline:       html.EscapeString(i.Firstline),
			Types:           sorted,
			FromDisplayName: getDisplayName(i),
			FromAddress:     getFromAddress(i),
			ReceivedDate:    getReceivedDate(i.Date),
			ToAddresses:     getToAddresses(i)})
	}
	return
}
