package list

import (
	"a.yandex-team.ru/mail/iex/taksa/client"
	"a.yandex-team.ru/mail/iex/taksa/daria"
	"a.yandex-team.ru/mail/iex/taksa/errs"
	"a.yandex-team.ru/mail/iex/taksa/hound"
	"a.yandex-team.ru/mail/iex/taksa/iex"
	"a.yandex-team.ru/mail/iex/taksa/logger"
	"a.yandex-team.ru/mail/iex/taksa/meta"
	"a.yandex-team.ru/mail/iex/taksa/request"
	"a.yandex-team.ru/mail/iex/taksa/search"
	"a.yandex-team.ru/mail/iex/taksa/widgets"
	"a.yandex-team.ru/mail/iex/taksa/widgets/common"
	"encoding/json"
	"fmt"
	"strconv"
	"time"
)

const (
	methodName string = "list"
)

func logError(l logger.Interface, msg string) {
	l.Error(methodName, msg)
}

func logErrorWithLossCount(l logger.Interface, msg string, count int) {
	l.ErrorExtra(methodName, msg, logger.Extra{"count": strconv.Itoa(count)})
}

type Method struct {
	Cfg  Config
	Wcfg widgets.Config
}

func getEnvelopes(version string, body []byte, l logger.Interface) (res []meta.Envelope, err error) {
	switch version {
	case "hound":
		res, err = hound.GetEnvelopes(body, l)
	case "daria_messages":
		res, err = daria.GetEnvelopes(body, l)
	case "search":
		res, err = search.GetEnvelopes(body, l)
	default:
		err = errs.BadRequest{Err: "invalid version specified"}
	}
	return
}

func getHeaders(r request.Interface) (headers map[string]string) {
	headers = map[string]string{}
	if xrid := r.GetHeader("X-Request-Id"); xrid != "" {
		headers["X-Request-Id"] = xrid
	}
	if ticket := r.GetHeader("X-Ya-Service-Ticket"); ticket != "" {
		headers["X-Ya-Service-Ticket"] = ticket
	}
	if expBoxes := r.GetHeader("X-Yandex-Enabledexpboxes"); expBoxes != "" {
		headers["X-Yandex-Enabledexpboxes"] = expBoxes
	}
	return headers
}

func getUser(r request.Interface) (user iex.User, err error) {
	uid := r.GetParam("uid")
	if uid == "" {
		err = errs.BadRequest{Err: "uid required"}
		return
	}
	user = iex.User{UID: uid, SUID: r.GetParam("suid"), MDB: r.GetParam("mdb")}
	return
}

func (method Method) GetTimeout() time.Duration {
	return time.Duration(method.Cfg.Timeout)
}

func logStats(l logger.Interface, requested, widgetable, facts, widgets int) {
	l.DebugExtra(methodName, "stats", logger.Extra{
		"requested":  strconv.Itoa(requested),
		"widgetable": strconv.Itoa(widgetable),
		"facts":      strconv.Itoa(facts),
		"widgets":    strconv.Itoa(widgets)})
}

func (method Method) Do(r request.Interface, l logger.Interface, c client.Interface) (response string, e error) {
	user, e := getUser(r)
	if e != nil {
		logError(l, "no uid in the request")
		return
	}
	envelopes, e := getEnvelopes(r.GetParam("version"), r.GetBody(), l)
	if e != nil {
		return
	}
	headers := getHeaders(r)
	var retry *string = nil
	if r.HasParam("retry") {
		retry = new(string)
		*retry = r.GetParam("retry")
	}
	sync := r.HasParam("sync")
	taksaHost := r.GetHeader("X-Forwarded-Host")

	widgets := widgets.Widgets{Cfg: method.Wcfg}
	envelopesFiltered := widgets.Widgetable(envelopes)
	res := common.Response{Widgets: []common.WidgetDto{}}
	if len(envelopesFiltered) != 0 {
		var extractor iex.Impl
		if sync {
			extractor = iex.Impl{Host: method.Cfg.SyncIexHost, Port: method.Cfg.SyncIexPort, Path: method.Cfg.IexPath, Log: l, Cli: c}
		} else {
			extractor = iex.Impl{Host: method.Cfg.IexHost, Port: method.Cfg.IexPort, Path: method.Cfg.IexPath, Log: l, Cli: c}
		}

		facts, err := extractor.Fetch(user, envelopesFiltered, headers, time.Duration(method.Cfg.IexTimeout), retry, sync, taksaHost)
		if err != nil {
			logStats(l, len(envelopes), len(envelopesFiltered), 0, 0)
			logErrorWithLossCount(l, "iex fetch failed", len(envelopesFiltered))
			e = errs.InternalError{Err: "iex fetch failed"}
			return
		}
		w, _ := widgets.Enrich(facts, l, c, r)
		res.Widgets = common.NewWidgetDtoSlice(w...)
		logStats(l, len(envelopes), len(envelopesFiltered), len(facts), len(w))
	}
	out, marshallingError := json.MarshalIndent(res, "", "\t")
	if marshallingError != nil {
		l.ErrorExtra(methodName, "cannot marshal response", logger.Extra{"err": marshallingError.Error()})
		e = errs.InternalError{Err: "response marchalling error"}
		return
	}
	response = string(out)
	l.Debug(methodName, fmt.Sprintf("%v widgets in response", len(res.Widgets)))
	return
}
