package search

import (
	"a.yandex-team.ru/mail/iex/taksa/errs"
	"a.yandex-team.ru/mail/iex/taksa/logger"
	"a.yandex-team.ru/mail/iex/taksa/meta"
	"a.yandex-team.ru/mail/iex/taksa/tutil"
	"encoding/json"
	"sort"
	"strings"
	"time"
)

type Response struct {
	Envelopes  []Message `json:"envelopes"`
	TopResults []Message `json:"top-relevant"`
}

func (response Response) valid() bool {
	if len(response.Envelopes) == 0 {
		return false
	}
	for _, i := range response.Envelopes {
		if !i.valid() {
			return false
		}
	}
	return true
}

type Message struct {
	Mid       string      `json:"mid"`
	Subject   string      `json:"subject"`
	Firstline string      `json:"firstline"`
	Types     []int       `json:"types"`
	From      []Recipient `json:"from"`
	To        []Recipient `json:"to"`
	Date      interface{} `json:"date"`
}

type Recipient struct {
	Local       string `json:"local"`
	Domain      string `json:"domain"`
	DisplayName string `json:"displayName"`
}

func (msg Message) valid() bool {
	return msg.Mid != ""
}

func getDisplayName(m Message) (res string) {
	if len(m.From) == 0 {
		return
	}
	res = m.From[0].DisplayName
	if res == "" {
		res = m.From[0].Local + "@" + m.From[0].Domain
	}
	return
}

func getFromAddress(m Message) (res string) {
	if len(m.From) > 0 {
		res = m.From[0].Local + "@" + m.From[0].Domain
	}
	return
}

func getToAddresses(m Message) (res []string) {
	if len(m.To) < 1 {
		return []string{}
	}
	for _, rcpt := range m.To {
		res = append(res, rcpt.Local+"@"+rcpt.Domain)
	}
	return
}

func getSortedTypes(m Message) (res sort.IntSlice) {
	res = sort.IntSlice(m.Types)
	res.Sort()
	return
}

func trimParenthesisAndWhitespace(orig []byte) []byte {
	str := string(orig)
	trimmed := strings.Trim(str, "() \t\r\n")
	return []byte(trimmed)
}

func getReceivedDate(i interface{}) (res time.Time) {
	res, err := tutil.GetTimeFromSomething(i)
	if err != nil {
		res = time.Now()
	}
	return
}

func GetEnvelopes(body []byte, l logger.Interface) (res []meta.Envelope, e error) {
	var response Response
	if err := json.Unmarshal(trimParenthesisAndWhitespace(body), &response); err != nil {
		l.ErrorExtra("list", "cannot parse body json", logger.Extra{"err": err.Error()})
		e = errs.BadRequest{Err: "bad json in body"}
		return
	}
	if !response.valid() {
		l.Error("list", "body json lacks mid")
		e = errs.BadRequest{Err: "invalid body"}
		return
	}
	for _, i := range append(response.Envelopes, response.TopResults...) {
		res = append(res, meta.Envelope{
			Mid:             i.Mid,
			Subject:         i.Subject,
			Firstline:       i.Firstline,
			Types:           getSortedTypes(i),
			FromDisplayName: getDisplayName(i),
			FromAddress:     getFromAddress(i),
			ReceivedDate:    getReceivedDate(i.Date),
			ToAddresses:     getToAddresses(i)})
	}
	return
}
