package server

import (
	"fmt"
	"net/http"
	"time"

	"a.yandex-team.ru/mail/iex/taksa/client"
	"a.yandex-team.ru/mail/iex/taksa/errs"
	"a.yandex-team.ru/mail/iex/taksa/future"
	"a.yandex-team.ru/mail/iex/taksa/logger"
	"a.yandex-team.ru/mail/iex/taksa/request"
)

type Method interface {
	Do(r request.Interface, l logger.Interface, c client.Interface) (response string, e error)
	GetTimeout() time.Duration
}

type NotFound struct {
}

func (NotFound) Do(r request.Interface, l logger.Interface, c client.Interface) (response string, e error) {
	e = errs.BadRequest{Err: "not found"}
	l.Error("server", "method not found")
	return
}

func (NotFound) GetTimeout() time.Duration {
	return 100 * time.Millisecond
}

func BindMethod(name string, method Method) {
	mux.HandleFunc("/api/"+name, MakeAPIHandler(method))
}

func MakeAPIHandler(method Method) http.HandlerFunc {
	return func(w http.ResponseWriter, r *http.Request) {
		apiHandler(w, r, method)
	}
}

func apiHandler(w http.ResponseWriter, r *http.Request, method Method) {
	start := time.Now()
	req := request.New(r)
	log := newAPILogger(req)
	sync := req.HasParam("sync")
	cli := client.New(log, sync)

	log.accept(req)
	code := http.StatusOK
	w.Header().Set("Content-Type", "application/json")

	functor := future.Functor(func() (interface{}, error) {
		return method.Do(req, log, cli)
	}).Async()
	var res interface{}
	var err error
	if sync {
		res, err = functor.Get()
	} else {
		res, err = functor.GetTimed(method.GetTimeout())
	}
	if err != nil {
		switch err.(type) {
		case errs.BadRequest:
			code = http.StatusBadRequest
		case errs.InternalError:
			code = http.StatusInternalServerError
		case future.TimeoutError:
			err = errs.InternalError{Err: "taksa timeout"}
			code = http.StatusInternalServerError
		}
		http.Error(w, err.Error(), code)
	} else {
		_, _ = fmt.Fprintln(w, res)
	}
	log.done(code, time.Since(start))
}
