package types

import (
	"sort"
	"strings"

	"a.yandex-team.ru/mail/iex/taksa/tutil"
)

type Types [][]int

const (
	Or  string = "|"
	And string = "&"
)

type BadTypes string

func (badTypes BadTypes) Error() string {
	return "bad types: " + string(badTypes)
}

func trimSpaceInStringSlice(original []string) (trimmed []string, err error) {
	for _, dirty := range original {
		clean := strings.TrimSpace(dirty)
		if len(clean) > 0 {
			trimmed = append(trimmed, clean)
		} else {
			return []string{}, BadTypes(dirty)
		}
	}
	return
}

func Parse(s string) (t Types, err error) {
	anyOf := strings.Split(s, Or)
	for _, oneOf := range anyOf {
		allOf, e := trimSpaceInStringSlice(strings.Split(oneOf, And))
		if e != nil {
			return Types(nil), e
		}
		ints, e := tutil.Strings2IntsStrict(allOf)
		if e != nil {
			return Types(nil), e
		}
		if len(ints) > 0 {
			t = append(t, ints)
		}
	}
	return
}

func (types *Types) UnmarshalJSON(buf []byte) (err error) {
	*types, err = Parse(strings.Trim(string(buf), "\""))
	return
}

func (types Types) String() string {
	var anyOf []string
	for _, allOf := range types {
		anyOf = append(anyOf, strings.Join(tutil.Ints2Strings(allOf), And))
	}
	prettyOr := " " + Or + " "
	return strings.Join(anyOf, prettyOr)
}

func (types Types) Fits(message sort.IntSlice) bool {
	for _, oneOf := range types {
		if tutil.Includes(message, oneOf) {
			return true
		}
	}
	return false
}
