package weather

import (
	"encoding/json"
	"fmt"
	"time"

	"a.yandex-team.ru/mail/iex/taksa/client"
	"a.yandex-team.ru/mail/iex/taksa/logger"
)

type Impl struct {
	Cfg Config
	Log logger.Interface
	Cli client.Interface
}

type Response struct {
	Forecasts []Day `json:"forecasts"`
}
type Day struct {
	Hours []Hour `json:"hours"`
}
type Hour struct {
	TS          int64 `json:"hour_ts"`
	Temperature int   `json:"temp"`
}

type Error string

func (err Error) Error() string {
	return fmt.Sprintf("weather error: [%s]", string(err))
}

func findTemperature(date time.Time, hours []Hour, precision time.Duration) (pos int, found bool) {
	date = date.Round(precision)
	for i, h := range hours {
		if time.Unix(h.TS, 0).Round(precision).Equal(date) {
			return i, true
		}
	}
	return -1, false
}

func flatten(forecasts []Day) (res []Hour) {
	for _, day := range forecasts {
		res = append(res, day.Hours...)
	}
	return
}

func (impl Impl) Get(geoid int, date time.Time) (res string, err error) {
	if geoid == 0 {
		err = Error("invalid geoid")
		impl.Log.Info("weather", "zero geoid")
		return
	}
	url := fmt.Sprintf("%v/v1/forecast?geoid=%v&limit=2", impl.Cfg.Host, geoid)
	headers := map[string]string{"Host": impl.Cfg.HeaderHost}
	headers["X-Yandex-Weather-Key"] = impl.Cfg.Key
	got, err := impl.Cli.Get(client.Params{URL: url, Headers: headers, Timeout: time.Duration(impl.Cfg.Timeout)})
	if err != nil {
		impl.Log.Error("weather", "http error")
		return
	}
	var response Response
	if err = json.Unmarshal([]byte(got), &response); err != nil {
		impl.Log.ErrorExtra("weather", "json parse error", logger.Extra{"err": err.Error()})
		return
	}
	flatHours := flatten(response.Forecasts)
	pos, found := findTemperature(date, flatHours, time.Hour)
	if found {
		res = fmt.Sprintf("%+d°C", flatHours[pos].Temperature)
	} else {
		err = Error("time not found")
		impl.Log.Debug("weather", "time not found")
	}
	return
}
