package avia

import (
	"a.yandex-team.ru/mail/iex/taksa/client"
	"a.yandex-team.ru/mail/iex/taksa/currency"
	"a.yandex-team.ru/mail/iex/taksa/iex"
	"a.yandex-team.ru/mail/iex/taksa/logger"
	"a.yandex-team.ru/mail/iex/taksa/request"
	"a.yandex-team.ru/mail/iex/taksa/tanker"
	"a.yandex-team.ru/mail/iex/taksa/tutil"
	"a.yandex-team.ru/mail/iex/taksa/weather"
	"a.yandex-team.ru/mail/iex/taksa/widgets/common"
	"time"
)

type Class struct {
	Cfg      Config
	Fact     iex.Fact
	Logger   logger.Interface
	Client   client.Interface
	Request  request.Request
	Weather  weather.Interface
	Currency currency.Interface
	Tanker   tanker.Interface
}

type Error string

func (e Error) Error() string {
	return "avia error: " + string(e)
}

func (Class) Name() string { return AviaWidget{}.Type() }

func (class Class) mid() string { return class.Fact.Envelope.Mid }

func (class Class) Extend() (common.Widget, error) {
	ticket, err := class.getTicket()
	if err != nil {
		class.Logger.ErrorExtra(class.Name(), "iex ticket convertion failed", logger.Extra{"err": err.Error()})
		return nil, err
	}

	class.Weather = weather.Impl{Cfg: class.Cfg.Weather, Log: class.Logger, Cli: class.Client}
	class.Currency = currency.Impl{Cfg: class.Cfg.Currency, Log: class.Logger, Cli: class.Client}

	subtype, expired := ticket.Subtype(), ticket.Expired(time.Now())
	departed := ticket.Departed(time.Now())
	isReminderSOType := tutil.ContainsInt(class.Fact.Envelope.Types, 63)
	isBoardingPassSOType := tutil.ContainsInt(class.Fact.Envelope.Types, 60)

	switch {

	case (subtype == "boardingpass" || isBoardingPassSOType) && !expired:
		return class.makeBoardingPassWidget(ticket)

	case (subtype == "boardingpass" || isBoardingPassSOType) && expired:
		return class.makeExpiredBoardingPassWidget(ticket)

	case subtype == "eticket" && !expired:
		return class.makeTicketWidget(ticket)

	case subtype == "booking-today" && !departed:
		return class.makeTomorrowWidget(ticket)

	case subtype == "booking-today" && departed && !expired:
		return class.makeTicketWidget(ticket)

	case (subtype == "eticket" || subtype == "booking-today") && expired:
		return class.makeExpiredWidget(ticket)

	case (subtype == "reminder" || isReminderSOType) && !expired:
		return class.makeReminderWidget(ticket)

	case (subtype == "reminder" || isReminderSOType) && expired:
		return nil, nil

	case subtype == "booking" || subtype == "canceling" ||
		subtype == "changing" || subtype == "undefined":
		return class.makeFactlessWidget(ticket)

	default:
		class.Logger.ErrorExtra(class.Name(), "unsupported subtype", logger.Extra{"subtype": subtype})
		return nil, Error("unsupported subtype " + subtype)
	}
}
