package avia

import (
	"fmt"
	"net/url"

	"a.yandex-team.ru/mail/iex/taksa/currency"
	"a.yandex-team.ru/mail/iex/taksa/logger"
	"a.yandex-team.ru/mail/iex/taksa/tanker"
	"a.yandex-team.ru/mail/iex/taksa/widgets/common"
)

func (class Class) getLogo(ticket Ticket, ct common.LogoColorF, icon common.LogoIconF) (c common.Control, err error) {
	iata, _ := ticket.AirlineIataCode()
	logo := common.Logo{Cfg: class.Cfg.Logo, Key: iata, DefaultName: class.Fact.Envelope.FromDisplayName}
	return common.MakeLogoControl(common.LogoName(logo), icon(logo), ct(logo), logo.Cfg.Zubchiki), nil
}

func (class Class) getTitle() (c common.Control, err error) {
	c = common.MakeTitleControl(class.Fact.Envelope.Subject)
	return
}

func (class Class) getBoardingPassTitle() (c common.Control, err error) {
	text, err := class.Tanker.GetString(tanker.BoardingPass)
	if err != nil {
		text = "Boarding pass"
	}
	c = common.MakeTitleControl(text)
	return
}

func (class Class) getRoute(ticket Ticket) (c common.Control, err error) {
	route := fillRoute(ticket, ticket.ArrivalCity, class.Tanker)
	if !route.canPrint() {
		err = Error("not enough info to print route")
		return
	}
	var str string
	if class.Request.GetMobile() {
		str = route.printNextFlightFirstLine()
	} else {
		str = route.printNextFlightInOneLine()
	}
	c = common.MakeTextControlWithHTMLEntities(common.Description1, str)
	return
}

func (class Class) getRouteAux(ticket Ticket) (c common.Control, err error) {
	route := fillRoute(ticket, ticket.ArrivalCity, class.Tanker)
	if !route.canPrint() {
		err = Error("not enough info to print route")
		return
	}
	str := ""
	if class.Request.GetMobile() {
		str = route.printNextFlightSecondLine()
	}
	c = common.MakeTextControlWithHTMLEntities(common.Description1Aux, str)
	return
}

func (class Class) getShortRoute(ticket Ticket) (c common.Control, err error) {
	route := fillRoute(ticket, ticket.LastArrivalCity, class.Tanker)
	if !route.canPrint() {
		err = Error("not enough info to print route")
		return
	}
	var str string
	if class.Request.GetMobile() {
		str = route.printWholeTripFirstLine()
	} else {
		str = route.printWholeTripInOneLine()
	}
	c = common.MakeTextControlWithHTMLEntities(common.Description1, str)
	return
}

func (class Class) getShortRouteAux(ticket Ticket) (c common.Control, err error) {
	route := fillRoute(ticket, ticket.LastArrivalCity, class.Tanker)
	if !route.canPrint() {
		err = Error("not enough info to print route")
		return
	}
	str := ""
	if class.Request.GetMobile() {
		str = route.printWholeTripSecondLine()
	}
	c = common.MakeTextControlWithHTMLEntities(common.Description1Aux, str)
	return
}

func (class Class) getBooking(ticket Ticket) (c common.Control, err error) {
	text, _ := class.Tanker.GetString(tanker.HotelBooking)
	where, err := ticket.ArrivalCityGeoID()
	if err != nil {
		class.Logger.Error(class.Name(), "need city_arr_geoid to get booking link")
		return
	}
	whenDate, e := ticket.ArrivalDate()
	if e != nil {
		whenDate, e = ticket.DepartureDate()
	}
	var whenQuery string
	if e == nil {
		whenQuery = fmt.Sprintf("&when=%v", whenDate.Format("2006-01-02"))
	}
	tld := class.Request.GetTld()
	value := fmt.Sprintf("%v.%v/search/rooms?to=%v%v", class.Cfg.TravelLink, tld, where, whenQuery)
	c = common.MakeLinkControl(common.Action1, text, value)
	return
}

func (class Class) getPrintOrLink(ticket Ticket) (c common.Control, err error) {
	parts := ticket.PrintParts()
	url, _ := ticket.URL()
	if len(parts) == 0 && url != "" {
		label, _ := class.Tanker.GetString(tanker.GoToOrder)
		return common.MakeLinkControl(common.Action2, label, url), nil
	}
	return common.GetPrint(common.Action2, ticket.PrintParts(), class.Tanker)
}

func (class Class) getWeather(ticket Ticket) (c common.Control, err error) {
	dst, err := ticket.ArrivalCityGeoID()
	if err != nil {
		class.Logger.ErrorExtra(class.Name(), "insufficient weather data: city_arr_geoid", logger.Extra{"err": err.Error()})
		return
	}
	ts, err := ticket.ArrivalDate()
	if err != nil {
		class.Logger.ErrorExtra(class.Name(), "insufficient weather data: date_arr", logger.Extra{"err": err.Error()})
		return
	}
	w, err := class.Weather.Get(dst, ts)
	if err != nil {
		class.Logger.Error(class.Name(), "weather get failed")
		return
	}
	c = common.MakeTextControl(common.Description2, w)
	return
}

func (class Class) getCurrency(ticket Ticket) (c common.Control, err error) {
	fromCountry, err := ticket.DapartureCountryGeoID()
	if err != nil {
		class.Logger.ErrorExtra(class.Name(), "insufficient currency data: from_country_geoid", logger.Extra{"err": err.Error()})
		return
	}
	toCountry, err := ticket.ArrivalCountryGeoID()
	if err != nil {
		class.Logger.ErrorExtra(class.Name(), "insufficient currency data: to_country_geoid", logger.Extra{"err": err.Error()})
		return
	}
	curr, err := class.Currency.Get(currency.CountryGeoID(fromCountry), currency.CountryGeoID(toCountry))
	if err != nil {
		class.Logger.Error(class.Name(), "currency get failed")
		return
	}
	c = common.MakeTextControl(common.Description3, curr)
	return
}

func (class Class) getCheckin(ticket Ticket) (c common.Control, err error) {
	return common.GetLink(common.Action3, ticket.CheckinURL, tanker.WebCheckIn, class.Tanker)
}

func (class Class) getTransfer(ticket Ticket) (c common.Control, err error) {
	var label, value string
	transfer, _ := ticket.Transfer()
	switch transfer {
	case "aeroexpress":
		label, err = class.Tanker.GetString(tanker.Aeroexpress)
		value = class.Cfg.AeroexpressLink
	case "ytaxi":
		label, err = class.Tanker.GetString(tanker.Taxi)
		value = class.Cfg.YTaxiLink
	case "taxi":
		where, _ := ticket.DapartureCity()
		label, err = class.Tanker.GetString(tanker.Taxi)
		value = fmt.Sprintf("%v.%v/search/?text=%v+%v",
			class.Cfg.SearchLink, class.Request.GetTld(), url.QueryEscape(label), url.QueryEscape(where))
	default:

		err = Error("no transfer available")
		class.Logger.Error(class.Name(), err.Error())
		return
	}
	c = common.MakeLinkControl(common.Action4, label, value)
	return
}

func (class Class) getPrintOrMoreTransfer(ticket Ticket) (c common.Control, err error) {
	parts := ticket.PrintParts()
	if len(parts) > 0 {
		return common.GetPrint(common.Action2, ticket.PrintParts(), class.Tanker)
	}
	if transfer, _ := ticket.Transfer(); transfer == "aeroexpress" {
		label, _ := class.Tanker.GetString(tanker.Taxi)
		value := class.Cfg.YTaxiLink
		c = common.MakeLinkControl(common.Action2, label, value)
	} else {
		err = Error("no transfer needed")
	}
	return
}

func (class Class) getGoToOrder(ticket Ticket) (c common.Control, err error) {
	if ticket.Subtype() != "booking" && ticket.Subtype() != "eticket" {
		err = Error("we do not go to order for subtypes other than eticket and booking")
		return
	}
	url, _ := ticket.URL()
	if url != "" {
		label, _ := class.Tanker.GetString(tanker.GoToOrder)
		return common.MakeLinkControl(common.Action2, label, url), nil
	} else {
		err = Error("no url")
	}
	return
}
