package avia

import (
	"a.yandex-team.ru/mail/iex/taksa/tutil"
	"a.yandex-team.ru/mail/iex/taksa/widgets/common"
	"time"
)

type TicketData struct {
	common.IexDict
}

type Ticket struct {
	data     TicketData
	lastDate *time.Time
}

func (class Class) getTicket() (res Ticket, err error) {
	for _, f := range class.Fact.IEX {
		w, ok := f.(map[string]interface{})
		if ok && w["taksa_widget_type_1234543456546"] == "ticket" {
			tickets, ok := w["ticket"].([]interface{})
			if ok && len(tickets) > 0 {
				if data, ok := tickets[0].(map[string]interface{}); ok {
					res.Init(data)
					return
				}
			}
		}
	}
	err = Error("no tickets in iex")
	return
}

func (ticket *Ticket) Init(data common.IexDict) {
	ticket.data = TicketData{data}
	ticket.lastDate = ticket.getLastDate()
}

func (ticket Ticket) getDate(name string) (time.Time, error) {
	return tutil.GetTimeFromRfcString(ticket.data.GetSomething(name))
}

func (ticket Ticket) getInt(name string) (int, error) {
	return tutil.GetIntFromSomething(ticket.data.GetSomething(name))
}

func (ticket Ticket) Subtype() (retval string) {
	retval, _ = ticket.data.GetString("widget_subtype")
	return
}

func (ticket Ticket) DepartureDate() (time.Time, error) {
	return ticket.getDate("date_dep_rfc")
}

func (ticket Ticket) ArrivalDate() (time.Time, error) {
	return ticket.getDate("date_arr_rfc")
}

func (ticket Ticket) BackDepartureDate() (time.Time, error) {
	return ticket.getDate("date_dep_back_rfc")
}

func (ticket Ticket) DapartureCity() (string, error) {
	return ticket.data.GetString("city_dep")
}

func (ticket Ticket) DapartureCountryGeoID() (int, error) {
	return ticket.getInt("from_country_geoid")
}

type whereGetter func() (string, error)

func (ticket Ticket) ArrivalCity() (string, error) {
	return ticket.data.GetString("city_arr")
}

func (ticket Ticket) LastArrivalCity() (city string, e error) {
	if city, e = ticket.data.GetString("city_dep_back"); e != nil {
		if city, e = ticket.data.GetString("city_arr_last"); e != nil {
			city, e = ticket.ArrivalCity()
		}
	}
	return
}

func (ticket Ticket) ArrivalCityGeoID() (int, error) {
	return ticket.getInt("city_arr_geoid")
}

func (ticket Ticket) ArrivalCountryGeoID() (int, error) {
	return ticket.getInt("to_country_geoid")
}

func (ticket Ticket) DapartureAirport() (string, error) {
	return ticket.data.GetString("airport_dep")
}

func (ticket Ticket) ArrivalAirport() (string, error) {
	return ticket.data.GetString("airport_arr")
}

func (ticket Ticket) AirlineIataCode() (string, error) {
	return ticket.data.GetString("airline_iata")
}

func (ticket Ticket) PrintParts() []string {
	return common.PrintParts(ticket.data.GetSomething("print_parts"))
}

func (ticket Ticket) URL() (string, error) {
	return ticket.data.GetString("url")
}

func (ticket Ticket) CheckinURL() (string, error) {
	return ticket.data.GetString("checkin_url")
}

func (ticket Ticket) Transfer() (string, error) {
	return ticket.data.GetString("transfer")
}

func (ticket Ticket) ExpireDate() *time.Time {
	return ticket.lastDate
}

func (ticket Ticket) Expired(now time.Time) bool {
	return ticket.ExpireDate() != nil && ticket.ExpireDate().Unix() <= now.Unix()
}

func (ticket Ticket) Departed(now time.Time) bool {
	depDate, _ := ticket.DepartureDate()
	return depDate.Unix() <= now.Unix()
}

func (ticket Ticket) getLastDate() *time.Time {
	fieldsPrioDesc := []string{
		"date_arr_back_rfc",
		"date_dep_back_rfc",
		"date_arr_last_rfc",
		"date_dep_last_rfc",
		"date_arr_rfc",
		"date_dep_rfc",
	}
	for _, f := range fieldsPrioDesc {
		date, err := ticket.getDate(f)
		if err == nil {
			return &date
		}
	}
	return nil
}
