package avia

import (
	"a.yandex-team.ru/mail/iex/taksa/tanker"
	"fmt"
	"time"
)

const (
	space string = " "
)

type timeFormatter func(time, date string) string

func timeOnly(time, _ string) string {
	return fmt.Sprintf("%v", time)
}
func dateTime(time, date string) string {
	return fmt.Sprintf("%v %v", date, time)
}
func timeDate(time, date string) string {
	return fmt.Sprintf("%v %v", time, date)
}

func printTime(t time.Time, frmt timeFormatter, tanker tanker.Interface) (res string, ok bool) {
	nilTime := time.Time{}
	if t != nilTime {
		time := t.Format("15:04")
		date, err := tanker.GetDate(t)
		ok = err == nil
		res = frmt(time, date)
	}
	return
}

func sameDay(t1, t2 time.Time) bool {
	return t1.Truncate(24 * time.Hour).Equal(t2.Truncate(24 * time.Hour))
}

func getTimeFormatter(t1, t2 time.Time) timeFormatter {
	if sameDay(t1, t2) {
		return timeOnly
	}
	return timeDate
}

type TimeAndErr struct {
	Time time.Time
	Err  error
}

func makeTimeAndErr(t time.Time, e error) TimeAndErr {
	return TimeAndErr{t, e}
}

type StringAndErr struct {
	Str string
	Err error
}

func makeStringAndErr(s string, e error) StringAndErr {
	return StringAndErr{s, e}
}

type Route struct {
	When         TimeAndErr
	WhenArr      TimeAndErr
	Back         TimeAndErr
	From         StringAndErr
	FromAirport  StringAndErr
	Where        StringAndErr
	WhereAirport StringAndErr
	IsReturn     bool
	Tanker       tanker.Interface
}

func fillRoute(ticket Ticket, where whereGetter, tnkr tanker.Interface) (route Route) {
	route.Tanker = tnkr
	route.When = makeTimeAndErr(ticket.DepartureDate())
	route.WhenArr = makeTimeAndErr(ticket.ArrivalDate())
	route.Back = makeTimeAndErr(ticket.BackDepartureDate())
	route.From = makeStringAndErr(ticket.DapartureCity())
	route.FromAirport = makeStringAndErr(ticket.DapartureAirport())
	route.Where = makeStringAndErr(where())
	route.WhereAirport = makeStringAndErr(ticket.ArrivalAirport())
	route.IsReturn = route.Back.Err == nil
	return
}

func (route Route) canPrint() bool {
	return route.When.Err == nil &&
		route.From.Err == nil &&
		route.Where.Err == nil
}

func (route Route) printWholeTripInOneLine() (res string) {
	when, ok := printTime(route.When.Time, dateTime, route.Tanker)
	if ok {
		res += when + space
	}

	res += route.From.Str + space

	if route.IsReturn {
		res += "&planes;" + space
	} else {
		res += "&plane;" + space
	}

	res += route.Where.Str

	if route.IsReturn {
		if secondDate, ok := printTime(route.Back.Time, timeDate, route.Tanker); ok {
			res += space + secondDate
		}
	} else if route.WhenArr.Err == nil {
		timeFmt := getTimeFormatter(route.When.Time, route.WhenArr.Time)
		if secondDate, ok := printTime(route.WhenArr.Time, timeFmt, route.Tanker); ok {
			res += space + secondDate
		}
	}
	return
}

func (route Route) printWholeTripFirstLine() (res string) {
	res = route.From.Str + space + "&ndash;" + space + route.Where.Str
	return
}

func (route Route) printWholeTripSecondLine() (res string) {
	if when, ok := printTime(route.When.Time, dateTime, route.Tanker); ok {
		res += when
	}

	if route.IsReturn {
		if secondDate, ok := printTime(route.Back.Time, dateTime, route.Tanker); ok {
			res += space + "&planes;" + space + secondDate
		}
	} else if route.WhenArr.Err == nil {
		if secondDate, ok := printTime(route.WhenArr.Time, dateTime, route.Tanker); ok {
			res += space + "&plane;" + space + secondDate
		}
	}
	return
}

func (route Route) printNextFlightInOneLine() (res string) {
	when, ok := printTime(route.When.Time, dateTime, route.Tanker)
	if ok {
		res += when + space
	}

	res += route.From.Str + space
	if route.FromAirport.Str != "" {
		res += "(" + route.FromAirport.Str + ")" + space
	}

	res += "&plane;" + space

	res += route.Where.Str
	if route.WhereAirport.Str != "" {
		res += space + "(" + route.WhereAirport.Str + ")"
	}

	if route.WhenArr.Err == nil {
		timeFmt := getTimeFormatter(route.When.Time, route.WhenArr.Time)
		if secondDate, ok := printTime(route.WhenArr.Time, timeFmt, route.Tanker); ok {
			res += space + secondDate
		}
	}
	return
}

func (route Route) printNextFlightFirstLine() (res string) {
	res = route.From.Str + space + "&ndash;" + space + route.Where.Str
	return
}

func (route Route) printNextFlightSecondLine() (res string) {
	if when, ok := printTime(route.When.Time, dateTime, route.Tanker); ok {
		res += when
	}

	if route.WhenArr.Err == nil {
		if secondDate, ok := printTime(route.WhenArr.Time, dateTime, route.Tanker); ok {
			res += space + "&plane;" + space + secondDate
		}
	}
	return
}
