package calendar

import (
	"a.yandex-team.ru/mail/iex/taksa/tutil"
	"a.yandex-team.ru/mail/iex/taksa/widgets/common"
	"time"
)

type Event struct {
	common.IexDict
}

func (class Class) getEvent() (res Event, err error) {
	for _, f := range class.Fact.IEX {
		dict, ok := f.(map[string]interface{})
		res = Event{dict}
		if ok && res.GetSomething("taksa_widget_type_1234543456546") == "event-ticket" &&
			(res.GetSomething("widget_subtype") == "calendar" || res.GetSomething("widget_subtype") == "calendar-updated") {
			return
		}
	}
	err = Error("no event in iex")
	return
}

func (event Event) People() (int, error) {
	return tutil.GetIntFromSomething(event.GetSomething("people"))
}

type IexParticipant struct {
	name     string
	decision string
	email    string
}

func (event Event) GetParticipants() (res []IexParticipant) {
	if arr, ok := event.GetSomething("attendees").([]interface{}); ok {
		for _, elem := range arr {
			if ifmap, ok := elem.(map[string]interface{}); ok {
				name, ok1 := ifmap["name"].(string)
				decision, ok2 := ifmap["decision"].(string)
				email, ok3 := ifmap["email"].(string)
				if ok1 && ok2 && ok3 {
					res = append(res, IexParticipant{name, decision, email})
				}
			}
		}
	}
	return
}

func (event Event) GetOrganizer() (IexParticipant, error) {
	if p, ok := event.GetSomething("organizer").(map[string]interface{}); ok {
		name, ok1 := p["name"].(string)
		decision, ok2 := p["decision"].(string)
		email, ok3 := p["email"].(string)
		if ok1 && ok2 && ok3 {
			return IexParticipant{name, decision, email}, nil
		}
	}
	return IexParticipant{}, Error("no organizer in iex")
}

func (event Event) Title() (string, error) {
	return event.GetNonEmptyString("title")
}

func (event Event) HasNoTitle() bool {
	_, e := event.GetNonEmptyString("title")
	return e != nil
}

func (event Event) IsCancelled() bool {
	if b, ok := event.GetSomething("isCancelled").(bool); ok {
		return b
	}
	return false
}

func (event Event) IsPast() bool {
	nilTime := time.Time{}
	last, _ := tutil.GetTimeFromRfcString(event.GetSomething("dateEventLastRepetition"))
	until, _ := tutil.GetTimeFromRfcString(event.GetSomething("dateEventRepeatUntil"))
	end, _ := tutil.GetTimeFromRfcString(event.GetSomething("end_date_rfc"))
	if last != nilTime {
		return last.Unix() < time.Now().Unix()
	} else if until != nilTime {
		return until.Unix() < time.Now().Unix()
	} else if end != nilTime {
		return end.Unix() < time.Now().Unix()
	}
	return false
}

func (event Event) Subtype() (retval string) {
	retval, _ = event.GetString("widget_subtype")
	return
}

func (event Event) IsUpdated() bool {
	val, _ := event.GetString("calendarMailType")
	return val == "event_update"
}

func (event Event) IsRepeatable() bool {
	val, e := event.GetString("repetitionDescription")
	if e == nil && val != "" {
		return true
	}
	if val2, ok := event.GetSomething("isRecurrence").(bool); ok {
		return val2
	}
	return false
}

func (event Event) GetRepetition() string {
	val, _ := event.GetString("repetitionDescription")
	return val
}

func (event Event) IsAllDay() bool {
	if b, ok := event.GetSomething("isAllDay").(bool); ok {
		return b
	}
	return false
}

func (event Event) Decision() (string, error) {
	return event.GetNonEmptyString("decision")
}

func (event Event) LookURL() (string, error) {
	return event.GetNonEmptyString("calendarUrl")
}

func (event Event) RecurrenceEventID() string {
	val, _ := event.GetString("recurrenceEventId")
	return val
}

func (event Event) ExternalEventID() string {
	val, _ := event.GetString("externalEventId")
	return val
}
