package common

import (
	"time"
)

type Widget interface {
	Mid() string
	Type() string
	SubType() string
	ExpireDate() *time.Time
	Valid() bool
	Double() bool
	ID() string
	Version() string
	Controls() []Control
	Init(string, []Control, *time.Time, string, string)
}

func Valid(w Widget) bool { return w.Valid() }

func Double(w Widget) bool { return w.Double() }

type ValidFalse struct{}

func (ValidFalse) Valid() bool { return false }

type ValidTrue struct{}

func (ValidTrue) Valid() bool { return true }

type SingleRowSize struct{}

func (SingleRowSize) Double() bool { return false }

type DoubleRowSize struct{}

func (DoubleRowSize) Double() bool { return true }

type WidgetBase struct {
	mid        string
	controls   []Control
	expireDate *time.Time
	id         string
	version    string
}

func (base WidgetBase) Mid() string            { return base.mid }
func (base WidgetBase) ExpireDate() *time.Time { return base.expireDate }
func (base WidgetBase) Controls() []Control    { return base.controls }
func (base WidgetBase) ID() string             { return base.id }
func (base WidgetBase) Version() string        { return base.version }
func (base *WidgetBase) Init(mid string, controls []Control, expireDate *time.Time, id string, version string) {
	base.mid, base.controls, base.expireDate, base.id, base.version = mid, controls, expireDate, id, version
}

type WidgetError string

func (error WidgetError) Error() string {
	return "widget error: " + string(error)
}

func InitWidget(w Widget, mid string, acc *Accumulator, date *time.Time) (Widget, error) {
	controls, err := acc.Result()
	if err != nil {
		return nil, err
	}
	w.Init(mid, controls, date, "", "")
	return w, nil
}

type SubType interface {
	SubType() string
}

type TicketSubType struct{}

func (TicketSubType) SubType() string { return "ticket" }

type TomorrowSubType struct{}

func (TomorrowSubType) SubType() string { return "tomorrow" }

type ReminderSubType struct{}

func (ReminderSubType) SubType() string { return "reminder" }

type BoardingPassSubType struct{}

func (BoardingPassSubType) SubType() string { return "boardingpass" }

type FactlessSubType struct{}

func (FactlessSubType) SubType() string { return "factless" }

type BookingSubType struct{}

func (BookingSubType) SubType() string { return "booking" }

type CancelingSubType struct{}

func (CancelingSubType) SubType() string { return "canceling" }

type CinemaSubType struct{}

func (CinemaSubType) SubType() string { return "cinema" }

type Response struct {
	Widgets []WidgetDto `json:"widgets"`
}

type WidgetDto struct {
	Info         Info      `json:"info"`
	ControlsList []Control `json:"controls"`
}

func (dto WidgetDto) Mid() string {
	return dto.Info.Mid
}

func (dto WidgetDto) Type() string {
	return dto.Info.Type
}

func (dto WidgetDto) SubType() string {
	return dto.Info.SubType
}

func (dto WidgetDto) ExpireDate() *time.Time {
	return dto.Info.ExpireDate
}

func (dto WidgetDto) Valid() bool {
	return dto.Info.Valid
}

func (dto WidgetDto) Double() bool {
	return dto.Info.Double
}

func (dto WidgetDto) ID() string {
	return dto.Info.ID
}

func (dto WidgetDto) Version() string {
	return dto.Info.Version
}

func (dto WidgetDto) Controls() []Control {
	return dto.ControlsList
}

func (dto *WidgetDto) Init(mid string, controls []Control, expireDate *time.Time, id string, version string) {
	dto.Info.Mid, dto.ControlsList, dto.Info.ExpireDate, dto.Info.ID, dto.Info.Version = mid, controls, expireDate, id, version
}

func NewWidgetDto(w Widget) (r WidgetDto) {
	r.Info.Mid = w.Mid()
	r.Info.Type = w.Type()
	r.Info.SubType = w.SubType()
	r.Info.ExpireDate = w.ExpireDate()
	r.Info.Valid = Valid(w)
	r.Info.Double = Double(w)
	r.Info.ID = w.ID()
	r.Info.Version = w.Version()
	r.ControlsList = w.Controls()
	return
}

func NewWidgetDtoSlice(widgets ...Widget) (r []WidgetDto) {
	for _, w := range widgets {
		r = append(r, NewWidgetDto(w))
	}
	return
}

type Info struct {
	Mid        string     `json:"mid"`
	Type       string     `json:"type"`
	SubType    string     `json:"subtype"`
	ExpireDate *time.Time `json:"expire_date,omitempty"`
	Valid      bool       `json:"valid"`
	Double     bool       `json:"double"`
	ID         string     `json:"id"`
	Version    string     `json:"version"`
}
