package common

import (
	"encoding/hex"
	"math"
)

const (
	white string = "#ffffff"
	black string = "#000000"
)

type Color struct {
	R uint8
	G uint8
	B uint8
}

func parse(str string) (res Color) {
	if len(str) == 0 {
		return
	}
	bites, err := hex.DecodeString(str[1:])
	if err != nil || len(bites) != 3 {
		return
	}
	res.R = bites[0]
	res.G = bites[1]
	res.B = bites[2]
	return
}

func someShit(x uint8) (res float64) {
	y := float64(x) / 255
	if y <= 0.03928 {
		res = y / 12.92
	} else {
		res = math.Pow(((y + 0.055) / 1.055), 2.4)
	}
	return
}

func luminance(colorString string) (res float64) {
	//https://www.w3.org/TR/2008/REC-WCAG20-20081211/#relativeluminancedef
	color := parse(colorString)
	R := someShit(color.R)
	G := someShit(color.G)
	B := someShit(color.B)
	res = 0.2126*R + 0.7152*G + 0.0722*B
	return
}

func contrast(color1, color2 string) (res float64) {
	//https://www.w3.org/TR/2008/REC-WCAG20-20081211/#contrast-ratiodef
	l1 := luminance(color1)
	l2 := luminance(color2)
	if l1 > l2 {
		res = (l1 + 0.05) / (l2 + 0.05)
	} else {
		res = (l2 + 0.05) / (l1 + 0.05)
	}
	return
}

func whiteOrBlack(color string) string {
	if color == "transparent" {
		return black
	}
	whiteContrast := contrast(white, color)
	blackContrast := contrast(black, color)
	if whiteContrast > blackContrast {
		return white
	} else {
		return black
	}
}

func darkOrLight(textColor, icon string) string {
	if icon == "" {
		return ""
	}
	if textColor == white {
		return icon + "-light"
	}
	return icon + "-dark"
}
