package common

import (
	"fmt"
)

type ControlType string

const (
	LogoT           ControlType = "logo"
	Text            ControlType = "text"
	Link            ControlType = "link"
	Print           ControlType = "print"
	Delete          ControlType = "delete"
	Compose         ControlType = "compose"
	ImgLink         ControlType = "imglink"
	Drop            ControlType = "drop"
	Button          ControlType = "button"
	Buttons         ControlType = "buttons"
	RichPreviewLink ControlType = "rich_preview_link"
	BigImageT       ControlType = "bigimage"
)

func MakeTitleControl(label string) (c Control) {
	c.Role = "title"
	c.Type = Text
	c.Attributes = Attributes{"label": label, "has_html_entities": false}
	return
}

func MakeTitleControlWithHTMLEntities(label string) (c Control) {
	c.Role = "title"
	c.Type = Text
	c.Attributes = Attributes{"label": label, "has_html_entities": true}
	return
}

func MakeLogoControl(label string, icon string, color string, zubchiki bool) (c Control) {
	c.Role = "logo"
	c.Type = LogoT
	textColor := whiteOrBlack(color)
	c.Attributes = Attributes{
		"label":             label,
		"icon":              darkOrLight(textColor, icon),
		"color":             color,
		"text_color":        textColor,
		"zubchiki":          zubchiki,
		"has_html_entities": false,
	}
	return
}

func MakeTextControl(role fmt.Stringer, label string) (c Control) {
	c.Role = role.String()
	c.Type = Text
	c.Attributes = Attributes{"label": label, "has_html_entities": false}
	return
}

func MakeTextControlWithHTMLEntities(role fmt.Stringer, label string) (c Control) {
	c.Role = role.String()
	c.Type = Text
	c.Attributes = Attributes{"label": label, "has_html_entities": true}
	return
}

func MakeLinkControl(role fmt.Stringer, label, value string) (c Control) {
	c.Role = role.String()
	c.Type = Link
	c.Attributes = Attributes{"label": label, "value": value, "has_html_entities": false}
	return
}

func MakePrintControl(role fmt.Stringer, label, value string) (c Control) {
	c.Role = role.String()
	c.Type = Print
	c.Attributes = Attributes{"label": label, "value": value, "has_html_entities": false}
	return
}

func MakeDeleteControl(role fmt.Stringer, label string) (c Control) {
	c.Role = role.String()
	c.Type = Delete
	c.Attributes = Attributes{"label": label, "has_html_entities": false}
	return
}

func MakeComposeControl(role fmt.Stringer, label, mid string, fixRcpt bool) (c Control) {
	c.Role = role.String()
	c.Type = Compose
	c.Attributes = Attributes{"label": label, "mid": mid, "fix_rcpt": fixRcpt, "has_html_entities": false}
	return
}

func MakeImgLinkControl(role fmt.Stringer, img, link string) (c Control) {
	c.Role = role.String()
	c.Type = ImgLink
	c.Attributes = Attributes{"label": img, "value": link, "has_html_entities": false}
	return
}

type Participant struct {
	Status  string
	Name    string
	Address string
}

func MakeDropControl(role fmt.Stringer, label string, participants []Participant) (c Control) {
	c.Role = role.String()
	c.Type = Drop
	c.Attributes = Attributes{"label": label, "value": participants, "has_html_entities": true}
	return
}

type SingleButton struct {
	Label             string
	Value             string
	Type              ControlType
	ExternalEventID   string
	RecurrenceEventID string
	GoToState         int
	Yellow            bool
}

type ButtonState struct {
	ID             int
	ButtonsInState []SingleButton
}

func MakeButtonsControl(role fmt.Stringer, buttons []ButtonState) (c Control) {
	c.Role = role.String()
	c.Type = Buttons
	c.Attributes = Attributes{"states": buttons, "has_html_entities": false}
	return
}

func MakeRichPreviewControl(attributes Attributes) (c Control) {
	c.Role = "rich_preview"
	c.Type = RichPreviewLink
	c.Attributes = attributes
	return
}

func MakeBigImageControl(value string, button string, image string) (c Control) {
	c.Role = "bigimage"
	c.Type = BigImageT
	c.Attributes = Attributes{"value": value, "button": button, "image": image}
	return
}
