package logic

import (
	. "a.yandex-team.ru/mail/imap-fuzzer/fuzz/util"
	. "context"
	"fmt"
	"log"
	"sync"
	"sync/atomic"
)

type Runner interface {
	Run(config Config, auth Auth) ([]ErrorCommand, bool, error)
	RunAllAsync(ctx Context, out chan<- ErrorCommand, configs []Config, auth Auth) (*sync.WaitGroup, map[string]*Progress)
}

type Progress struct {
	Done uint64
	Left uint64
}

func NewRunner() Runner {
	return RunnerImpl{}
}

type RunnerImpl struct {
}

func (r RunnerImpl) Run(config Config, auth Auth) ([]ErrorCommand, bool, error) {
	output := make([]ErrorCommand, 0)
	scenario, scenarioErr := NewScenario("imap.yandex.ru:993", auth, config)
	if scenarioErr != nil {
		return []ErrorCommand{}, false, nil
	}

	if errCmd, err := scenario.Run(); errCmd != nil {
		output = append(output, *errCmd)
	} else if err != nil {
		return output, len(output) > 0, err
	}

	return output, len(output) > 0, nil
}

func (r RunnerImpl) RunAllAsync(ctx Context, out chan<- ErrorCommand, configs []Config, auth Auth) (*sync.WaitGroup, map[string]*Progress) {
	var wg sync.WaitGroup

	var progress = make(map[string]*Progress)

	for _, cfg := range configs {
		for i := uint64(0); i < cfg.Instances; i++ {
			wg.Add(1)

			var cp *Progress
			var ok bool

			cp, ok = progress[cfg.Name]

			if !ok {
				cp = &Progress{
					Done: 0,
					Left: 0,
				}
				progress[cfg.Name] = cp
			}

			cp.Left += cfg.Attempts

			callback := func(c *uint64) func() {
				return func() {
					atomic.AddUint64(c, 1)
				}
			}(&cp.Done)

			go r.wrap(ctx, out, cfg, auth, &wg, callback)
		}
	}

	return &wg, progress
}

func (r RunnerImpl) wrap(ctx Context, out chan<- ErrorCommand, config Config, auth Auth, wg *sync.WaitGroup, onAttemptEnd func()) {
	defer wg.Done()

	for i := uint64(0); i < config.Attempts; i++ {
		select {
		case <-ctx.Done():
			return
		default:
		}

		arr, notEmpty, err := r.Run(config, auth)
		if notEmpty {
			for _, errCmd := range arr {
				out <- errCmd
			}
		}
		onAttemptEnd()

		if err != nil {
			log.Println("stop scenario with error: " + fmt.Sprintf("%v", err))
			return
		}
	}
}
