package logic

import (
	"a.yandex-team.ru/mail/imap-fuzzer/fuzz"
	. "a.yandex-team.ru/mail/imap-fuzzer/fuzz/operations"
	. "a.yandex-team.ru/mail/imap-fuzzer/fuzz/types"
	. "a.yandex-team.ru/mail/imap-fuzzer/fuzz/util"
	imap4 "a.yandex-team.ru/mail/imap-fuzzer/imap"
	"encoding/base64"
	"encoding/json"
	"errors"
	"fmt"
	"log"
	"strings"
	"time"
)

type clientCreator = func() (*imap4.Client, error)

type Scenario struct {
	name        string
	login       Login
	commands    []Command
	newClient   clientCreator
	config      Config
	randomState RandState
	needAuth    bool
	needSelect  bool
}

type ErrorCommand struct {
	Name  string `json:"scenarioName"`
	State string `json:"state"`
	Msg   string `json:"errMsg"`
	PCmd  string `json:"PrintableCommand"`
	Cmd   string `json:"Command"`
	PCont string `json:"PrintableContinue"`
	Cont  string `json:"Continue"`
}

func newErrorCommand(name string, err error, rs RandState, cmd Command) *ErrorCommand {
	return &ErrorCommand{
		Name:  name,
		State: base64.StdEncoding.EncodeToString(rs.GetState()),
		Msg:   err.Error(),
		PCmd:  strings.ToValidUTF8(cmd.AsString(), "?"),
		Cmd:   base64.StdEncoding.EncodeToString([]byte(cmd.AsString())),
		Cont:  base64.StdEncoding.EncodeToString([]byte(strings.Join(cmd.ContinueLines(), "\n"))),
		PCont: strings.ToValidUTF8(strings.Join(cmd.ContinueLines(), "[\\n]"), "?"),
	}
}

func (e ErrorCommand) Error() string {
	if b, err := json.Marshal(e); err == nil {
		return string(b)
	}
	panic("can't serialize error")
}

func (scenario *Scenario) Run() (*ErrorCommand, error) {
	// т.к. клиенты могут пересоздаваться - удобнее закрывать руками, чем через defer
	pclient, err := scenario.newClient()
	if err != nil {
		return nil, err
	}

	for _, cmd := range scenario.commands {
		bye, err := pclient.Execute(cmd)
		if bye {
			log.Println("Preparing client again (reason: server said: 'bye')")
			_ = pclient.Close()
			pclient, err = scenario.newClient()
			if err != nil {
				return nil, err
			}
		}
		if err != nil {
			return newErrorCommand(scenario.name, err, scenario.randomState, cmd), nil
		}
	}
	_ = pclient.Close()
	return nil, nil
}

func generateCommands(ctx fuzz.Context, rs RandState, config Config) []Command {
	result := make([]Command, config.Len)
	for i := range result {
		result[i] = nextCommand(ctx, rs, config.Dist)
	}
	return result
}

func nextCommand(ctx fuzz.Context, rs RandState, dist CommandDist) Command {
	initializerName := dist.Peek(rs)
	initializer := AllInitializers[initializerName]
	cmd := initializer(ctx, rs).Command()
	fuzzed := cmd.Fuzz(rs)
	return *fuzzed
}

func prepareContext(ctx *fuzz.Context, client *imap4.Client) (err error) {
	login := Login{
		Username: ctx.Username,
		Password: ctx.Password,
	}

	_, err = client.ExecuteWithConsumer(login.Command(), ctx.TryApply)
	if err != nil {
		return
	}

	s := SelectInbox

	_, err = client.ExecuteWithConsumer(s.Command(), ctx.TryApply)
	if err != nil {
		return
		//panic("can't prepare context: " + fmt.Sprintf("%v", err))
	}

	_, err = client.ExecuteWithConsumer(Command{
		Tag:       BasicStringArg("INIT-001"),
		Name:      "LIST",
		Arguments: CmdArguments{BasicStringArg("\"\""), BasicStringArg("*")},
	}, ctx.TryApply)

	return
}

func newPreparedClient(host string, ctx fuzz.Context, timeout time.Duration, attempts uint) (*imap4.Client, error) {
	for i := uint(0); i < attempts; i++ {
		createClient := func() (*imap4.Client, error) {
			return imap4.DialTLS(host, nil, timeout)
		}

		pclient, err := createClient()
		if err != nil {
			return nil, err
		}

		if err := prepareContext(&ctx, pclient); err != nil {
			if pclient != nil {
				_ = pclient.Close()
			}
			log.Println("can't prepare context: " + fmt.Sprintf("%v", err))
			continue
		}
		return pclient, nil
	}
	return nil, errors.New("cant create prepared client")
}

func NewScenario(host string, auth Auth, config Config) (*Scenario, error) {
	rs := NewRandState(config.StateLen)
	authData := auth.Any()
	username := Email{Name: authData.Login, Domain: "yandex", Zone: "ru"}
	password := Password{BasicStringArg: BasicStringArg(authData.Password)}

	ctx := fuzz.NewContext(username, password)

	newClient := func() (client *imap4.Client, e error) {
		return newPreparedClient(host, ctx, time.Minute, 3)
	}

	if c, err := newClient(); err == nil {
		//noinspection GoUnhandledErrorResult
		defer c.Close()
		if err := prepareContext(&ctx, c); err != nil {
			return nil, err
		}
	} else {
		return nil, err
	}

	return &Scenario{
		name: config.Name,
		login: Login{
			Username: username,
			Password: password,
		},
		commands:    generateCommands(ctx, rs, config),
		newClient:   newClient,
		config:      config,
		randomState: rs,
		needAuth:    true,
		needSelect:  true,
	}, nil
}
