package types

import (
	"math"
	"strconv"
	"sync"
	"time"

	. "a.yandex-team.ru/mail/imap-fuzzer/fuzz/util"
)

type Date string

var DateDistribution = CreateDistribution(map[string]uint{
	"no-changes":        5,
	"any-valid":         45,
	"insert-number":     10,
	"replace-substring": 10,
	"any-string":        1,
})

var (
	dateLayoutsBuildOnce sync.Once
	dateLayouts          []string
)

func (d Date) AsString() string {
	return string(d)
}

func (d Date) Fuzz(r RandState) CmdArgument {
	switch DateDistribution.Peek(r) {
	case "no-changes":
		return d
	case "any-string":
		return Date(r.AnyString())
	case "any-valid":
		return Date(AnyValidTime(r))
	case "replace-substring":
		s := d.AsString()
		a, b := r.RandInt("rml", len(s)), r.RandInt("rmr", len(s))
		min := int(math.Min(float64(a), float64(b)))
		max := int(math.Max(float64(a), float64(b)))
		insertion := r.AnyString()
		l := r.RandInt("ins-len", 10)
		if len(insertion) < l {
			l = len(insertion)
		}
		insertion = insertion[:l]
		return Date(s[:min] + insertion + s[max:])
	case "insert-number":
		s := d.AsString()
		idx := r.RandInt("replace-num", len(s))
		return Date(s[:idx] + strconv.Itoa(r.RandInt("any-number", 12)) + s[idx:])
	default:
		panic("unexpected case")
	}
}

func AnyValidTime(r RandState) string {
	dateLayoutsBuildOnce.Do(buildDateLayouts)
	idx := r.RandInt("time-layout", len(dateLayouts))
	t := time.Unix(r.Int64("time"), r.Int64("time-ns"))
	return t.Format(dateLayouts[idx])
}

func buildDateLayouts() {
	// Generate layouts based on RFC 5322, section 3.3.

	dows := [...]string{"", "Mon, "}   // day-of-week
	days := [...]string{"2", "02"}     // day = 1*2DIGIT
	years := [...]string{"2006", "06"} // year = 4*DIGIT / 2*DIGIT
	seconds := [...]string{":05", ""}  // second
	// "-0700 (MST)" is not in RFC 5322, but is common.
	zones := [...]string{"-0700", "MST", "-0700 (MST)"} // zone = (("+" / "-") 4DIGIT) / "GMT" / ...

	for _, dow := range dows {
		for _, day := range days {
			for _, year := range years {
				for _, second := range seconds {
					for _, zone := range zones {
						s := dow + day + " Jan " + year + " 15:04" + second + " " + zone
						dateLayouts = append(dateLayouts, s)
					}
				}
			}
		}
	}

	dateLayouts = append(dateLayouts, time.ANSIC,
		time.UnixDate,
		time.RubyDate,
		time.RFC822,
		time.RFC822Z,
		time.RFC850,
		time.RFC1123,
		time.RFC1123Z,
		time.RFC3339,
		time.RFC3339Nano,
		time.Kitchen,
		time.Stamp,
		time.StampMilli,
		time.StampMicro,
		time.StampNano,
	)
}
