package types

import (
	. "a.yandex-team.ru/mail/imap-fuzzer/fuzz/util"
	"strings"
)

type MailBox struct {
	BasicStringArg
}

var Inbox = MailBox{BasicStringArg: "INBOX"}

func (mb MailBox) Parts() []string {
	return strings.Split(mb.BasicStringArg.AsString(), "|")
}

func MailBoxFromParts(parts []string) MailBox {
	return MailBox{
		BasicStringArg: BasicStringArg(strings.Join(parts, "|")),
	}
}

var MailBoxFuzzDistribution = CreateDistribution(map[string]uint{
	"fuzz-as-string":  10,
	"append-folder":   25,
	"append-folders":  1,
	"append-valid":    25,
	"append-valids":   25,
	"append-single":   1,
	"shuffle-folders": 25,
	"insert":          5,
	"change-|":        1,
	"dont-change":     3,
	"peek-first":      5,
	"truncate":        5,
})

func (mb MailBox) Fuzz(r RandState) CmdArgument {

	switch MailBoxFuzzDistribution.Peek(r) {
	case "fuzz-as-string":
		return mb.BasicStringArg.Fuzz(r)

	case "append-folder":
		return appendAnyFolders(mb.BasicStringArg, 1, r)

	case "append-folders":
		count := r.RandInt("folders-to-append", 32)
		return appendAnyFolders(mb.BasicStringArg, count, r)

	case "append-valid":
		return appendFolders(mb.BasicStringArg, 1, Alnum, r)

	case "append-valids":
		count := r.RandInt("folders-to-append-valid", 32)
		return appendFolders(mb.BasicStringArg, count, Alnum, r)

	case "append-single":
		count := r.RandInt("folders-to-append-valid", 128)
		to := mb.BasicStringArg
		for i := 0; i < count; i++ {
			to = to + BasicStringArg("|"+string(LetterRunes[r.RandInt("peek-rune", len(LetterRunes))]))
		}
		return to

	case "shuffle-folders":
		return BasicStringArg(
			strings.Join(
				r.ShuffleString(
					"shuffle-folders",
					strings.Split(string(mb.BasicStringArg), "|")),
				"|"))

	case "change-|":
		return BasicStringArg(
			strings.ReplaceAll(
				string(mb.BasicStringArg),
				"|",
				string(r.FastRandBytes(r.RandInt("any-bytes", 8)))))

	case "insert":
		parts := strings.Split(string(mb.BasicStringArg), "|")
		pos := r.RandInt("insert-pos", len(parts))
		parts = append(parts, "")
		copy(parts[pos+1:], parts[pos:])
		parts[pos] = r.AnyString()
		return BasicStringArg(strings.Join(parts, "|"))

	case "dont-change":
		return mb

	case "peek-first":
		first := strings.Split(string(mb.BasicStringArg), "|")[0]
		return BasicStringArg(first)

	case "truncate":
		parts := strings.Split(string(mb.BasicStringArg), "|")
		parts = parts[:r.RandInt("truncate", len(parts))]
		return BasicStringArg(strings.Join(parts, "|"))

	default:
		panic("missing case")
	}
}

func appendAnyFolders(to BasicStringArg, count int, rs RandState) BasicStringArg {
	return appendFolders(to, count, StringType(rs.RandInt("st", int(LastStringTypeItem))), rs)
}

func appendFolders(to BasicStringArg, count int, t StringType, rs RandState) BasicStringArg {
	for i := 0; i < count; i++ {
		to = to + BasicStringArg("|"+rs.AnyTypedString(t))
	}
	return to
}
