package types

import (
	. "a.yandex-team.ru/mail/imap-fuzzer/fuzz/util"
	"strconv"
	"strings"
)

type CriteriaProducer = func(r RandState, data CriteriaData) SingleCriteria

type CriteriaData struct {
	User  Email
	Flags []Flag
}

var producers []CriteriaProducer

func init() {
	producers = []CriteriaProducer{
		AnyValidUnarLogicCriteria,
		AnyValidSearchFlag,
		AnyValidStringCriteria,
		AnyValidDateCriteria,
		AnyValidFlagKeywordCriteria,
		AnyValidOrLogicCriteria,
		AnySetIdCriteria,
	}
}

func AnyCriteria(r RandState, data CriteriaData) SingleCriteria {
	p := producers[r.RandInt("producer-idx", len(producers))]
	return p(r, data)
}

func AnySetIdCriteria(r RandState, data CriteriaData) SingleCriteria {
	if r.RandInt("use-parts", 100) < 15 {
		parts := make([]string, 0)
		count := r.RandInt("parts-count", 8)
		for i := 0; i < count; i++ {
			parts = append(parts, AnySetIdCriteria(r, data).AsString())
		}
		return SingleCriteria{
			Name: "",
			Arg:  ConstantText(strings.Join(parts, ",")),
		}
	} else {
		switch IdSetDistribution.Peek(r) {
		case "number":
			return SingleCriteria{
				Name: "",
				Arg:  argnum(r, -1),
			}
		case "pnumber":
			return SingleCriteria{
				Name: "",
				Arg:  argnum(r, 1000000),
			}
		case "2numbers":
			return SingleCriteria{
				Name: "",
				Arg:  argnum(r, 1000) + ":" + argnum(r, 1000000),
			}
		case "asteriks":
			return SingleCriteria{
				Name: "",
				Arg:  argnum(r, 1000) + ":*",
			}
		default:
			panic("unreachable case")
		}
	}
}

func argnum(rs RandState, limit int) BasicStringArg {
	if limit < 0 {
		return BasicStringArg(strconv.Itoa(int(rs.Int64("any-int"))))
	}
	return BasicStringArg(strconv.Itoa(rs.RandInt("any-int", limit)))
}

func AnyValidOrLogicCriteria(r RandState, data CriteriaData) SingleCriteria {
	return SingleCriteria{
		Name: "OR",
		Arg:  CmdArguments{AnyCriteria(r, data), AnyCriteria(r, data)},
	}
}

func AnyValidUnarLogicCriteria(r RandState, data CriteriaData) SingleCriteria {
	op, _ := r.PeekAnyString("ulogic-name", ValidUnarLogicNames)
	return SingleCriteria{
		Name: op,
		Arg:  AnyCriteria(r, data),
	}
}

func AnyValidSearchFlag(r RandState, data CriteriaData) SingleCriteria {
	f, _ := r.PeekAnyString("search-flag", ValidSearchFlags)
	return SingleCriteria{
		Name: "",
		Arg:  BasicStringArg(f),
	}
}

func AnyValidStringCriteria(r RandState, data CriteriaData) SingleCriteria {
	name, _ := r.PeekAnyString("string-criteria", ValidStringCriteriaNames)
	var body CmdArgument
	if r.Bool("criteria-use-email") {
		body = data.User
	} else {
		body = BasicStringArg(r.AnyTypedString(Alnum))
	}
	return SingleCriteria{
		Name: name,
		Arg:  body,
	}
}

func AnyValidDateCriteria(r RandState, data CriteriaData) SingleCriteria {
	name, _ := r.PeekAnyString("date-criteria", ValidDateCriteriaNames)
	return SingleCriteria{
		Name: name,
		Arg:  Date(AnyValidTime(r)),
	}
}

func AnyValidFlagKeywordCriteria(r RandState, data CriteriaData) SingleCriteria {
	name, _ := r.PeekAnyString("flag-keyword-criteria", ValidFlagKeywords)
	flag := data.Flags[r.RandInt("flag-keyword", len(data.Flags))]
	return SingleCriteria{
		Name: name,
		Arg:  flag,
	}
}

var ValidUnarLogicNames = []string{
	"NOT",
	"AND",
}

var ValidFlagKeywords = []string{
	"KEYWORD",
	"UNKEYWORD",
}

var ValidSearchFlags = []string{
	"ALL",
	"ANSWERED",
	"DELETED",
	"FLAGGED",
	"NEW",
	"OLD",
	"RECENT",
	"SEEN",
	"UNANSWERED",
	"UNDELETED",
	"UNFLAGGED",
	"UNSEEN",
	"DRAFT",
	"UNDRAFT",
}

var ValidDateCriteriaNames = []string{
	"BEFORE",
	"ON",
	"SINCE",
	"SENTBEFORE",
	"SENTON",
	"SENTSINCE",
}

var ValidStringCriteriaNames = []string{
	"BCC",
	"BODY",
	"CC",
	"FROM",
	"SUBJECT",
	"TEXT",
	"TO",
}

var IdSetDistribution = CreateDistribution(map[string]uint{
	"number":   10,
	"pnumber":  100,
	"2numbers": 25,
	"asteriks": 25,
})

type SingleCriteria struct {
	Name string
	Arg  CmdArgument
}

func (s SingleCriteria) AsString() string {
	if len(s.Name) > 0 {
		return s.Name + " " + s.Arg.AsString()
	}
	return s.Arg.AsString()
}

func (s SingleCriteria) Fuzz(r RandState) CmdArgument {
	name := s.Name
	if r.BoolUniq("fuzz-arg-name") {
		name = FuzzString(name, r)
	}
	return SingleCriteria{
		Name: name,
		Arg:  s.Arg.Fuzz(r),
	}
}
