package types

import (
	"a.yandex-team.ru/mail/imap-fuzzer/fuzz/naughty"
	. "a.yandex-team.ru/mail/imap-fuzzer/fuzz/util"
	"math"
)

var specialRunes = []string{"±", "!", "@", "#", "$", "%", "^", "&", "*", "(", ")", "_", "+", "=", "-", "\"", "№", "%", ":", ",", ".", ";", "(", ")", "<", ">", "`", "~", "|", "/", "\\", "\t", "\r", "..", "../", "./", "~/", " ", ""}

type StringFuzzType int

var StringDistribution = CreateDistribution(map[string]uint{
	"no-changes":        2,
	"shuffle":           5,
	"remove-last":       5,
	"remove-first":      5,
	"remove-med":        5,
	"return-empty":      1,
	"random-alnum":      10,
	"random-noctrl":     10,
	"random-noctrl-ex":  10,
	"random-noctrl-ex2": 10,
	"insert-rune":       5,
	"insert-byte-pair":  5,
	"insert-special":    5,
	"naughty":           25,
	"fuzzed-naughty":    10,
})

func FuzzString(s string, r RandState) (fuzzed string) {
	switch StringDistribution.Peek(r) {
	case "no-changes":
		return s

	case "shuffle":
		return string(r.Shuffle("shuffle-string", []byte(s)))

	case "remove-last":
		return s[r.RandInt("rl", len(s)):]

	case "remove-first":
		return s[:r.RandInt("rf", len(s))]

	case "remove-med":
		a, b := r.RandInt("rml", len(s)), r.RandInt("rmr", len(s))
		min := int(math.Min(float64(a), float64(b)))
		max := int(math.Max(float64(a), float64(b)))
		return s[min:max]

	case "return-empty":
		return ""

	case "random-alnum":
		return r.AnyTypedString(Alnum)

	case "random-noctrl":
		return r.AnyTypedString(NoCtrl)

	case "random-noctrl-ex":
		return r.AnyTypedString(NoCtrlAndExt)

	case "random-noctrl-ex2":
		return r.AnyTypedString(NoCtrlAndExt2)

	case "unicode-range":
		return r.AnyTypedString(UnicodeRange)

	case "insert-rune":
		pos := r.RandInt("rune-pos", len(s))
		return s[:pos] + string(rune(r.Int64("some-rune"))) + s[pos:]

	case "insert-byte-pair":
		pos := r.RandInt("byte-pair-pos", len(s))
		return s[:pos] + string(r.FastRandBytes(2)) + s[pos:]

	case "insert-special":
		pos := r.RandInt("special-pos", len(s))
		sp, _ := r.PeekAnyString("special", specialRunes)
		return s[:pos] + sp + s[pos:]

	case "fuzzed-naughty":
		ns, _ := r.PeekAnyString("naughty-string", naughty.Strings)
		return ns

	case "naughty":
		ns, _ := r.PeekAnyString("naughty-string", naughty.Strings)
		return FuzzString(ns, r)

	default:
		panic("unreachable case")
	}
}

//func RandString(n int, r util.RandState) string {
//	result := string(r.FastRandBytes(n))
//	result = strings.ReplaceAll(result, " ", "")
//	result = strings.ReplaceAll(result, "\r", "")
//	result = strings.ReplaceAll(result, "\n", "")
//	return result
//}
