package util

import (
	"encoding/binary"
	"fmt"
	"hash"
	"hash/fnv"
	"math"
	"math/rand"
	"strconv"
	"time"
)

var LetterRunes = []rune("abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ1234567890_-")

type StringType int

const (
	Any StringType = iota
	Alnum
	NoCtrl
	NoCtrlAndExt
	NoCtrlAndExt2
	UnicodeRange
	LastStringTypeItem
)

var AnySizeDistribution = CreateDistribution(map[string]uint{
	"extra-small": 100,
	"small":       100,
	"med":         50,
	"big":         25,
	"large":       10,
	"extra-large": 1,
})

type RandState interface {
	Bool(name string) bool
	BoolUniq(name string) bool
	Int64(name string) int64
	RandInt(name string, max int) int
	Shuffle(name string, data []byte) []byte
	ShuffleString(name string, data []string) []string
	Perm(name string, len int) []int
	FastRandInt(max int) int
	FastRandByte() byte
	FastRandBytes(n int) []byte
	AnyString() string
	AnyTypedString(t StringType) string
	PeekAnyString(name string, arr []string) (string, error)
	Float64(name string) float64

	GetState() []byte
}

type RandStateImpl struct {
	n     int
	state []byte
	hash  hash.Hash32
	rand  *rand.Rand
}

func (r *RandStateImpl) GetState() []byte {
	b := make([]byte, len(r.state))
	copy(b, r.state)
	return b
}

func (r *RandStateImpl) Perm(name string, n int) []int {
	r.rand.Seed(r.Int64(name))
	return r.rand.Perm(n)
}

func (r *RandStateImpl) ShuffleString(name string, data []string) []string {
	dest := make([]string, len(data))
	perm := r.Perm(name, len(data))
	for i, v := range perm {
		dest[v] = data[i]
	}
	return dest
}

func (r *RandStateImpl) PeekAnyString(name string, arr []string) (string, error) {
	if len(arr) == 0 {
		return "", fmt.Errorf("can't peek from em[ty array")
	}
	return arr[r.RandInt(name, len(arr))], nil
}

func (r *RandStateImpl) Float64(name string) float64 {
	r.rand.Seed(r.Int64(name))
	n := r.rand.Float64()

	if r.BoolUniq(name + "-adding") {
		n = n * float64(r.Int64(name+"-adding"))
	}

	if r.Bool(name + "-sign") {
		n = n * -1
	}
	return n
}

type RandStateHolder struct {
	*RandStateImpl
}

func NewRandState(stateLen uint64) RandState {
	rnd := rand.New(rand.NewSource(time.Now().UnixNano()))
	state := make([]byte, stateLen)
	rnd.Read(state)

	r := RandStateImpl{
		n:     0,
		state: state,
		hash:  fnv.New32a(),
		rand:  rnd,
	}

	return RandStateHolder{RandStateImpl: &r}
}

func (r *RandStateImpl) AnyString() string {
	return r.AnyTypedString(StringType(r.RandInt("chose-type", int(LastStringTypeItem))))
}

func (r *RandStateImpl) AnyTypedString(t StringType) string {
	var size int
	switch AnySizeDistribution.Peek(r) {
	case "extra-small":
		size = r.RandInt("small-str", 32)
	case "small":
		size = r.RandInt("small-str", 128)
	case "med":
		size = 64 + r.RandInt("med-str", 256)
	case "big":
		size = 256 + r.RandInt("big-str", 1024)
	case "large":
		size = 1024 + r.RandInt("large-str", 16*1024)
	case "extra-large":
		size = 4096 + r.RandInt("ex-large-str", 2*1024*1024)
	}

	switch t {
	case Any:
		return r.anyStringRaw("any-string", size)
	case Alnum:
		return r.alnumStringRaw("alnum-string", size)
	case NoCtrl:
		return StripCtlFromUTF8(r.anyStringRaw("no-ctrl-string", size))
	case NoCtrlAndExt:
		return StripCtlAndExtFromUTF8(r.anyStringRaw("no-ctrl-string", size))
	case NoCtrlAndExt2:
		return StripCtlAndExtFromUnicode(r.anyStringRaw("no-ctrl-string", size))
	case UnicodeRange:
		return r.UnicodeRangeString("unicode-range-string", size)
	default:
		panic("can't choose string type WTF?!")

	}
}

func (r *RandStateImpl) anyStringRaw(name string, n int) string {
	return string(r.FastRandBytes(n))
}

func (r *RandStateImpl) alnumStringRaw(name string, n int) string {
	bytes := r.FastRandBytes(n)
	b := make([]rune, n)
	for i := range b {
		b[i] = LetterRunes[int(bytes[i])%len(LetterRunes)]
	}
	return string(b)
}

func (r *RandStateImpl) FastRandBytes(n int) []byte {
	result := make([]byte, n)
	//todo vkokarev fix it?
	start := r.nextIndex("bytes")
	for i := range result {
		result[i] = byte(int(r.state[(start+i)%len(r.state)]) * r.n)
	}
	return result
}

func (r *RandStateImpl) Bool(name string) bool {
	return r.next(name) <= 127
}

func (r *RandStateImpl) BoolUniq(name string) bool {
	return r.next(name) == 127
}

func (r *RandStateImpl) Int64(name string) int64 {
	return int64(binary.LittleEndian.Uint64([]byte{r.next(name), r.next(name), r.next(name), r.next(name), r.next(name), r.next(name), r.next(name), r.next(name)}))
}

func (r *RandStateImpl) RandInt(name string, max int) int {
	if max == 0 {
		return 0
	}
	n := binary.LittleEndian.Uint64([]byte{r.next(name), r.next(name), r.next(name), r.next(name), r.next(name), r.next(name), r.next(name), r.next(name)})
	return int(n % uint64(max))
}

func (r *RandStateImpl) FastRandInt(max int) int {
	return r.RandInt("fast-random", max)
}

func (r *RandStateImpl) FastRandByte() byte {
	return r.next("rand-byte")
}

func (r *RandStateImpl) next(name string) byte {
	result := r.state[r.nextIndex(name)]
	return result
}

func (r *RandStateImpl) Shuffle(name string, data []byte) []byte {
	r.rand.Seed(r.Int64(name))
	dest := make([]byte, len(data))
	perm := r.rand.Perm(len(data))
	for i, v := range perm {
		dest[v] = data[i]
	}
	return dest
}

func (r *RandStateImpl) nextIndex(name string) int {
	_, err := r.hash.Write([]byte(name + strconv.Itoa(r.n)))
	r.n++
	defer r.hash.Reset()

	if err != nil {
		panic("can't apply hash")
	}

	idx := int(math.Abs(float64(r.hash.Sum32()))) % len(r.state)
	return idx

}

type charRange struct {
	first, last rune
}

var unicodeRanges = []charRange{
	{' ', '~'},           // ASCII characters
	{'\u00a0', '\u02af'}, // Multi-byte encoded characters
	{'\u4e00', '\u9fff'}, // Common CJK (even longer encodings)
}

func (r *RandStateImpl) UnicodeRangeString(name string, n int) string {
	runes := make([]rune, n)
	for i := range runes {
		urange := unicodeRanges[r.RandInt("unicode-ranges", len(unicodeRanges))]
		count := int(urange.last - urange.first)
		runes[i] = urange.first + rune(r.RandInt("u-rune", count))
	}
	return string(runes)
}
