package imap4

import (
	"a.yandex-team.ru/mail/imap-fuzzer/fuzz/operations"
	"errors"
	"log"
	"net"
	"os"
	"strconv"
	"strings"
	"time"
)

type Client struct {
	conn        *Conn
	isTLS       bool
	serverName  string
	errorLog    *log.Logger
	readTimeout time.Duration
}

type Consumer = func(line string) bool

type RespHandler = func(s string)

func NewClient(conn net.Conn, readTimeout time.Duration) (*Client, error) {
	w := NewClientWriter(nil)
	r := NewReader(nil)

	c := &Client{
		conn:        NewConn(conn, r, w),
		errorLog:    log.New(os.Stderr, "imap/client: ", log.LstdFlags),
		readTimeout: readTimeout,
	}

	return c, nil
}

func (c *Client) Close() error {
	return c.conn.Close()
}

func (c *Client) ExecuteWithConsumer(cmd operations.Command, consumer Consumer) (isBye bool, err error) {
	var bytesMore int
	//var outline string
	bytesMore, isBye, err, _ = c.executeLine(cmd.Tag.AsString(), cmd.AsString(), consumer)

	continueLines := cmd.ContinueLines()
	for i := 0; bytesMore > 0 && !isBye && err == nil; i++ {
		if len(continueLines) == 0 {
			return true, errors.New("got unexpected continue msg")
		}
		line := continueLines[i%len(continueLines)]
		bytesMore, isBye, err, _ = c.executeLine("", line, consumer)
	}
	return
}

func (c *Client) Execute(cmd operations.Command) (isBye bool, err error) {
	return c.ExecuteWithConsumer(cmd, func(line string) bool {
		//println("s:", line)
		return false
	})
}

func (c *Client) executeLine(tag string, line string, consumer Consumer) (bytesMore int, isBye bool, err error, outline string) {
	//println("c:", line)
	err = c.conn.writeStringCrlf(line)
	if err != nil {
		return
	}

	bytesMore, isBye, err, outline = c.processResponse(tag, c.readTimeout, func(line string) bool {
		return consumer(line)
	})

	if err != nil || isBye {
		return
	}

	return
}

func (c *Client) processResponse(tag string, timeout time.Duration, consumer Consumer) (bytesMore int, wasClosed bool, err error, line string) {
	for {
		//var line string
		if connErr := c.conn.SetReadDeadline(time.Now().Add(timeout)); connErr != nil {
			return 0, false, connErr, ""
		}

		line, err = c.conn.ReadLine()
		if err != nil {
			return
		}

		line = strings.Replace(line, tag, "", 1)
		line = strings.TrimSuffix(line, string(lf))
		line = strings.TrimSuffix(line, string(cr))
		line = strings.TrimLeft(line, " ")

		consumer(line)

		if strings.Contains(line, "* BYE") {
			return 0, true, nil, line
		}

		if strings.HasPrefix(line, "+") {
			// + go ahead with 10 octets
			n := strings.Split(line, " ")[4]
			bytesMore, _ = strconv.Atoi(n)
			return
		} else if strings.HasPrefix(line, "*") {
			continue
		} else {
			return
		}
	}
}
