package imap4

import (
	"crypto/tls"
	"net"
	"time"
)

type Dialer interface {
	// Dial connects to the given address.
	Dial(network, addr string) (net.Conn, error)
}

func DialTLS(addr string, tlsConfig *tls.Config, readTimeout time.Duration) (*Client, error) {
	return DialWithDialerTLS(new(net.Dialer), addr, tlsConfig, readTimeout)
}

func DialWithDialerTLS(dialer Dialer, addr string, tlsConfig *tls.Config, readTimeout time.Duration) (*Client, error) {
	conn, err := dialer.Dial("tcp", addr)
	if err != nil {
		return nil, err
	}

	serverName, _, _ := net.SplitHostPort(addr)
	if tlsConfig == nil {
		tlsConfig = &tls.Config{}
	}
	if tlsConfig.ServerName == "" {
		tlsConfig = tlsConfig.Clone()
		tlsConfig.ServerName = serverName
	}
	tlsConn := tls.Client(conn, tlsConfig)

	// We don't return to the caller until we try to receive a greeting. As such,
	// there is no way to set the client's Timeout for that action. As a
	// workaround, if the dialer has a timeout set, use that for the connection's
	// deadline.
	if netDialer, ok := dialer.(*net.Dialer); ok && netDialer.Timeout > 0 {
		err := tlsConn.SetDeadline(time.Now().Add(netDialer.Timeout))
		if err != nil {
			return nil, err
		}
	}

	c, err := NewClient(tlsConn, readTimeout)
	if err != nil {
		return nil, err
	}

	c.isTLS = true
	c.serverName = serverName

	if _, err := c.conn.ReadInfo(); err != nil {
		return nil, err
	}

	return c, nil
}
